<?php
/**
 * DatabaseCleaner - Bereinigt die Datenbank vor dem Backup
 * 
 * Entfernt unnötige Daten wie:
 * - Post-Revisionen
 * - Auto-Drafts
 * - Gelöschte Posts (Trash)
 * - Spam-Kommentare
 * - Gelöschte Kommentare
 * - Transients (abgelaufene und alle)
 * - Orphaned Postmeta
 * - Orphaned Commentmeta
 * - Plugin-Logs
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Backup;

if (!defined('ABSPATH')) {
    exit;
}

class DatabaseCleaner {
    
    /** @var array Statistiken */
    private $stats = [];
    
    /** @var bool Ob ein Snapshot vor der Bereinigung erstellt werden soll */
    private $create_snapshot = true;
    
    /** @var array Konfiguration */
    private $config;
    
    /**
     * Erstellt einen neuen DatabaseCleaner
     * 
     * @param array $config Konfiguration
     */
    public function __construct(array $config = []) {
        $this->config = array_merge([
            'clean_revisions' => true,
            'clean_auto_drafts' => true,
            'clean_trash' => true,
            'clean_spam_comments' => true,
            'clean_trash_comments' => true,
            'clean_transients' => true,
            'clean_orphaned_postmeta' => true,
            'clean_orphaned_commentmeta' => true,
            'clean_orphaned_termmeta' => true,
            'clean_expired_transients_only' => false, // true = nur abgelaufene
            'max_revisions_to_keep' => 0, // 0 = alle löschen
            'create_snapshot' => true,
        ], $config);
        
        $this->create_snapshot = $this->config['create_snapshot'];
    }
    
    /**
     * Führt die komplette Bereinigung durch
     * 
     * @return array Statistiken
     */
    public function clean(): array {
        global $wpdb;
        
        $this->stats = [
            'started_at' => current_time('mysql'),
            'snapshot_created' => false,
            'items_cleaned' => [],
            'space_freed' => 0,
            'errors' => [],
        ];
        
        // Optional: Snapshot erstellen
        if ($this->create_snapshot) {
            $this->createSnapshot();
        }
        
        // Bereinigungen durchführen
        if ($this->config['clean_revisions']) {
            $this->cleanRevisions();
        }
        
        if ($this->config['clean_auto_drafts']) {
            $this->cleanAutoDrafts();
        }
        
        if ($this->config['clean_trash']) {
            $this->cleanTrash();
        }
        
        if ($this->config['clean_spam_comments']) {
            $this->cleanSpamComments();
        }
        
        if ($this->config['clean_trash_comments']) {
            $this->cleanTrashComments();
        }
        
        if ($this->config['clean_transients']) {
            $this->cleanTransients();
        }
        
        if ($this->config['clean_orphaned_postmeta']) {
            $this->cleanOrphanedPostmeta();
        }
        
        if ($this->config['clean_orphaned_commentmeta']) {
            $this->cleanOrphanedCommentmeta();
        }
        
        if ($this->config['clean_orphaned_termmeta']) {
            $this->cleanOrphanedTermmeta();
        }
        
        // Tabellen optimieren
        $this->optimizeTables();
        
        $this->stats['completed_at'] = current_time('mysql');
        
        return $this->stats;
    }
    
    /**
     * Bereinigt nur einen bestimmten Bereich
     * 
     * @param string $area Bereich (z.B. 'revisions', 'transients')
     * @return int Anzahl gelöschter Einträge
     */
    public function cleanArea(string $area): int {
        switch ($area) {
            case 'revisions':
                return $this->cleanRevisions();
            case 'auto_drafts':
                return $this->cleanAutoDrafts();
            case 'trash':
                return $this->cleanTrash();
            case 'spam_comments':
                return $this->cleanSpamComments();
            case 'trash_comments':
                return $this->cleanTrashComments();
            case 'transients':
                return $this->cleanTransients();
            case 'orphaned_postmeta':
                return $this->cleanOrphanedPostmeta();
            case 'orphaned_commentmeta':
                return $this->cleanOrphanedCommentmeta();
            case 'orphaned_termmeta':
                return $this->cleanOrphanedTermmeta();
            default:
                return 0;
        }
    }
    
    /**
     * Gibt eine Vorschau der zu bereinigenden Daten
     * 
     * @return array Vorschau
     */
    public function preview(): array {
        global $wpdb;
        
        $preview = [];
        
        // Revisionen
        $preview['revisions'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'revision'"
        );
        
        // Auto-Drafts
        $preview['auto_drafts'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'auto-draft'"
        );
        
        // Papierkorb
        $preview['trash'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'trash'"
        );
        
        // Spam-Kommentare
        $preview['spam_comments'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved = 'spam'"
        );
        
        // Papierkorb-Kommentare
        $preview['trash_comments'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved = 'trash'"
        );
        
        // Transients
        if ($this->config['clean_expired_transients_only']) {
            $preview['transients'] = (int) $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->options} 
                 WHERE option_name LIKE '_transient_timeout_%' 
                 AND option_value < UNIX_TIMESTAMP()"
            );
        } else {
            $preview['transients'] = (int) $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->options} 
                 WHERE option_name LIKE '_transient_%' 
                 OR option_name LIKE '_site_transient_%'"
            );
        }
        
        // Orphaned Postmeta
        $preview['orphaned_postmeta'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} pm 
             LEFT JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
             WHERE p.ID IS NULL"
        );
        
        // Orphaned Commentmeta
        $preview['orphaned_commentmeta'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->commentmeta} cm 
             LEFT JOIN {$wpdb->comments} c ON cm.comment_id = c.comment_ID 
             WHERE c.comment_ID IS NULL"
        );
        
        // Orphaned Termmeta
        $preview['orphaned_termmeta'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->termmeta} tm 
             LEFT JOIN {$wpdb->terms} t ON tm.term_id = t.term_id 
             WHERE t.term_id IS NULL"
        );
        
        // Gesamtanzahl
        $preview['total'] = array_sum($preview);
        
        return $preview;
    }
    
    // ========================================
    // BEREINIGUNGS-METHODEN
    // ========================================
    
    /**
     * Löscht Post-Revisionen
     */
    private function cleanRevisions(): int {
        global $wpdb;
        
        $keep = $this->config['max_revisions_to_keep'];
        
        if ($keep > 0) {
            // Behalte X Revisionen pro Post
            // Komplexere Query, die neueste X behält
            $deleted = 0;
            
            $posts_with_revisions = $wpdb->get_col(
                "SELECT DISTINCT post_parent FROM {$wpdb->posts} 
                 WHERE post_type = 'revision' AND post_parent > 0"
            );
            
            foreach ($posts_with_revisions as $post_id) {
                $revisions = $wpdb->get_col($wpdb->prepare(
                    "SELECT ID FROM {$wpdb->posts} 
                     WHERE post_type = 'revision' AND post_parent = %d 
                     ORDER BY post_date DESC",
                    $post_id
                ));
                
                // Alle außer den neuesten X löschen
                $to_delete = array_slice($revisions, $keep);
                
                foreach ($to_delete as $rev_id) {
                    wp_delete_post_revision($rev_id);
                    $deleted++;
                }
            }
        } else {
            // Alle Revisionen löschen
            $revisions = $wpdb->get_col(
                "SELECT ID FROM {$wpdb->posts} WHERE post_type = 'revision'"
            );
            
            $deleted = 0;
            foreach ($revisions as $rev_id) {
                wp_delete_post_revision($rev_id);
                $deleted++;
            }
        }
        
        $this->stats['items_cleaned']['revisions'] = $deleted;
        return $deleted;
    }
    
    /**
     * Löscht Auto-Drafts
     */
    private function cleanAutoDrafts(): int {
        global $wpdb;
        
        $auto_drafts = $wpdb->get_col(
            "SELECT ID FROM {$wpdb->posts} WHERE post_status = 'auto-draft'"
        );
        
        $deleted = 0;
        foreach ($auto_drafts as $post_id) {
            wp_delete_post($post_id, true);
            $deleted++;
        }
        
        $this->stats['items_cleaned']['auto_drafts'] = $deleted;
        return $deleted;
    }
    
    /**
     * Leert den Papierkorb
     */
    private function cleanTrash(): int {
        global $wpdb;
        
        $trashed = $wpdb->get_col(
            "SELECT ID FROM {$wpdb->posts} WHERE post_status = 'trash'"
        );
        
        $deleted = 0;
        foreach ($trashed as $post_id) {
            wp_delete_post($post_id, true);
            $deleted++;
        }
        
        $this->stats['items_cleaned']['trash'] = $deleted;
        return $deleted;
    }
    
    /**
     * Löscht Spam-Kommentare
     */
    private function cleanSpamComments(): int {
        global $wpdb;
        
        $spam = $wpdb->get_col(
            "SELECT comment_ID FROM {$wpdb->comments} WHERE comment_approved = 'spam'"
        );
        
        $deleted = 0;
        foreach ($spam as $comment_id) {
            wp_delete_comment($comment_id, true);
            $deleted++;
        }
        
        $this->stats['items_cleaned']['spam_comments'] = $deleted;
        return $deleted;
    }
    
    /**
     * Löscht Papierkorb-Kommentare
     */
    private function cleanTrashComments(): int {
        global $wpdb;
        
        $trashed = $wpdb->get_col(
            "SELECT comment_ID FROM {$wpdb->comments} WHERE comment_approved = 'trash'"
        );
        
        $deleted = 0;
        foreach ($trashed as $comment_id) {
            wp_delete_comment($comment_id, true);
            $deleted++;
        }
        
        $this->stats['items_cleaned']['trash_comments'] = $deleted;
        return $deleted;
    }
    
    /**
     * Löscht Transients
     */
    private function cleanTransients(): int {
        global $wpdb;
        
        if ($this->config['clean_expired_transients_only']) {
            // Nur abgelaufene
            delete_expired_transients(true);
            $deleted = $wpdb->rows_affected ?? 0;
        } else {
            // Alle Transients
            $deleted = $wpdb->query(
                "DELETE FROM {$wpdb->options} 
                 WHERE option_name LIKE '_transient_%' 
                 OR option_name LIKE '_site_transient_%'"
            );
        }
        
        $this->stats['items_cleaned']['transients'] = $deleted;
        return $deleted;
    }
    
    /**
     * Löscht verwaiste Postmeta
     */
    private function cleanOrphanedPostmeta(): int {
        global $wpdb;
        
        $deleted = $wpdb->query(
            "DELETE pm FROM {$wpdb->postmeta} pm 
             LEFT JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
             WHERE p.ID IS NULL"
        );
        
        $this->stats['items_cleaned']['orphaned_postmeta'] = $deleted;
        return $deleted;
    }
    
    /**
     * Löscht verwaiste Commentmeta
     */
    private function cleanOrphanedCommentmeta(): int {
        global $wpdb;
        
        $deleted = $wpdb->query(
            "DELETE cm FROM {$wpdb->commentmeta} cm 
             LEFT JOIN {$wpdb->comments} c ON cm.comment_id = c.comment_ID 
             WHERE c.comment_ID IS NULL"
        );
        
        $this->stats['items_cleaned']['orphaned_commentmeta'] = $deleted;
        return $deleted;
    }
    
    /**
     * Löscht verwaiste Termmeta
     */
    private function cleanOrphanedTermmeta(): int {
        global $wpdb;
        
        $deleted = $wpdb->query(
            "DELETE tm FROM {$wpdb->termmeta} tm 
             LEFT JOIN {$wpdb->terms} t ON tm.term_id = t.term_id 
             WHERE t.term_id IS NULL"
        );
        
        $this->stats['items_cleaned']['orphaned_termmeta'] = $deleted;
        return $deleted;
    }
    
    /**
     * Optimiert Datenbank-Tabellen
     */
    private function optimizeTables(): void {
        global $wpdb;
        
        $tables = $wpdb->get_col("SHOW TABLES");
        
        foreach ($tables as $table) {
            $wpdb->query("OPTIMIZE TABLE `$table`");
        }
        
        $this->stats['tables_optimized'] = count($tables);
    }
    
    /**
     * Erstellt einen Snapshot vor der Bereinigung
     */
    private function createSnapshot(): void {
        // Einfacher Snapshot: Zähle aktuelle Zeilen
        global $wpdb;
        
        $this->stats['snapshot'] = [
            'posts' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts}"),
            'postmeta' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->postmeta}"),
            'comments' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->comments}"),
            'options' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->options}"),
        ];
        
        $this->stats['snapshot_created'] = true;
    }
    
    /**
     * Getter für Statistiken
     */
    public function getStats(): array {
        return $this->stats;
    }
}

