<?php
/**
 * FileRestorer - Datei-Wiederherstellung
 * 
 * Stellt Dateien aus dem Backup wieder her mit:
 * - Chunk-basierter Extraktion
 * - Berechtigungserhaltung
 * - Konflikterkennung
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Restore;

use JenvaBackupMigration\Core\Container;

if (!defined('ABSPATH')) {
    exit;
}

class FileRestorer {
    
    /** @var callable|null Progress-Callback */
    private $progress_callback;
    
    /**
     * Setzt den Progress-Callback
     */
    public function setProgressCallback(callable $callback): void {
        $this->progress_callback = $callback;
    }
    
    /**
     * Stellt Uploads wieder her
     */
    public function restoreUploads(Container $container): array {
        $upload_dir = wp_upload_dir();
        $destination = $upload_dir['basedir'];
        
        return $this->restoreSegmentType($container, 'uploads', $destination);
    }
    
    /**
     * Stellt Themes wieder her
     */
    public function restoreThemes(Container $container): array {
        $destination = get_theme_root();
        
        return $this->restoreSegmentType($container, 'themes', $destination);
    }
    
    /**
     * Stellt Plugins wieder her
     */
    public function restorePlugins(Container $container): array {
        $destination = WP_PLUGIN_DIR;
        
        return $this->restoreSegmentType($container, 'plugins', $destination);
    }
    
    /**
     * Stellt Segmente eines Typs wieder her
     */
    private function restoreSegmentType(Container $container, string $type, string $destination): array {
        $result = [
            'type' => $type,
            'destination' => $destination,
            'files_restored' => 0,
            'errors' => [],
        ];
        
        try {
            $files_restored = $container->extractType($type, $destination, function($done, $total, $file) {
                if ($this->progress_callback) {
                    call_user_func($this->progress_callback, $done, $total, $file);
                }
            });
            
            $result['files_restored'] = $files_restored;
        } catch (\Exception $e) {
            $result['errors'][] = $e->getMessage();
        }
        
        return $result;
    }
}

