<?php
namespace JBM;

if (!defined('ABSPATH')) {
    exit;
}

class Schedule {
    
    public function __construct() {
        add_action('jbm_scheduled_backup_1', [$this, 'run_scheduled_backup_1']);
        add_action('jbm_scheduled_backup_2', [$this, 'run_scheduled_backup_2']);
    }
    
    /**
     * Aktualisiert einen Backup-Zeitplan (1 oder 2)
     */
    public function update_schedule($schedule_num, $schedule_data) {
        // Alten Zeitplan löschen
        $this->clear_schedule($schedule_num);
        
        if (!$schedule_data['enabled']) {
            return ['success' => true, 'message' => 'Zeitplan deaktiviert'];
        }
        
        $frequency = $schedule_data['frequency'];
        $timestamp = time();
        
        // Zeitpunkt berechnen basierend auf Frequenz
        if ($frequency === 'hourly') {
            // Stündlich: nächste volle Stunde
            $timestamp = strtotime('+1 hour', strtotime(date('Y-m-d H:00:00')));
        } else {
            // Zeit parsen
            $time_parts = explode(':', $schedule_data['time']);
            $hour = intval($time_parts[0]);
            $minute = isset($time_parts[1]) ? intval($time_parts[1]) : 0;
            
            if ($frequency === 'daily') {
                // Täglich: heute oder morgen zur angegebenen Zeit
                $timestamp = strtotime("today {$hour}:{$minute}:00");
                if ($timestamp < time()) {
                    $timestamp = strtotime("tomorrow {$hour}:{$minute}:00");
                }
                
            } elseif ($frequency === 'weekly') {
                // Wöchentlich: am angegebenen Wochentag
                $weekday = intval($schedule_data['weekday']);
                $weekday_names = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
                $target_day = $weekday_names[$weekday];
                
                // Nächster Auftritt dieses Wochentags
                $timestamp = strtotime("next {$target_day} {$hour}:{$minute}:00");
                
                // Wenn heute dieser Wochentag ist und die Zeit noch nicht vorbei
                if (date('w') == $weekday) {
                    $today_time = strtotime("today {$hour}:{$minute}:00");
                    if ($today_time > time()) {
                        $timestamp = $today_time;
                    }
                }
                
            } elseif ($frequency === 'monthly') {
                // Monatlich: am angegebenen Tag des Monats
                $monthday = intval($schedule_data['monthday']);
                $current_month = date('Y-m');
                $timestamp = strtotime("{$current_month}-{$monthday} {$hour}:{$minute}:00");
                
                // Wenn Tag in diesem Monat schon vorbei, nächster Monat
                if ($timestamp < time()) {
                    $next_month = date('Y-m', strtotime('+1 month'));
                    $timestamp = strtotime("{$next_month}-{$monthday} {$hour}:{$minute}:00");
                }
            }
        }
        
        // Ereignis planen
        $hook_name = 'jbm_scheduled_backup_' . $schedule_num;
        
        // WordPress übergibt Args direkt an die Callback-Funktion
        $success = wp_schedule_event($timestamp, $frequency, $hook_name, [
            $schedule_data['backup_type'],  // Arg 1: backup_type
            $schedule_num,                   // Arg 2: schedule_num
            $schedule_data                   // Arg 3: schedule_data
        ]);
        
        if ($success === false) {
            return ['success' => false, 'message' => 'Fehler beim Planen des Backups'];
        }
        
        return [
            'success' => true,
            'message' => 'Zeitplan erfolgreich gespeichert',
            'next_run' => date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $timestamp)
        ];
    }
    
    /**
     * Löscht einen geplanten Zeitplan
     */
    public function clear_schedule($schedule_num) {
        $hook_name = 'jbm_scheduled_backup_' . $schedule_num;
        $timestamp = wp_next_scheduled($hook_name);
        
        if ($timestamp) {
            wp_unschedule_event($timestamp, $hook_name);
        }
        
        wp_clear_scheduled_hook($hook_name);
    }
    
    /**
     * Führt geplantes Backup 1 aus
     * WordPress übergibt Args direkt als Parameter, nicht als Array!
     */
    public function run_scheduled_backup_1($backup_type = 'full', $schedule_num = 1, $schedule_data = []) {
        $logger = new Logger();
        $logger->log("🔔 Cron-Hook gefeuert: jbm_scheduled_backup_1");
        $logger->log("Parameter: backup_type={$backup_type}, schedule_num={$schedule_num}");
        
        // Falls der erste Parameter ein Array ist (alte Übergabe)
        if (is_array($backup_type)) {
            $args = $backup_type;
            $backup_type = isset($args['backup_type']) ? $args['backup_type'] : 'full';
            $schedule_num = isset($args['schedule_num']) ? $args['schedule_num'] : 1;
            $schedule_data = isset($args['schedule_data']) ? $args['schedule_data'] : [];
        }
        
        return $this->execute_backup($backup_type, 1, $schedule_data);
    }
    
    /**
     * Führt geplantes Backup 2 aus
     */
    public function run_scheduled_backup_2($backup_type = 'full', $schedule_num = 2, $schedule_data = []) {
        $logger = new Logger();
        $logger->log("🔔 Cron-Hook gefeuert: jbm_scheduled_backup_2");
        $logger->log("Parameter: backup_type={$backup_type}, schedule_num={$schedule_num}");
        
        // Falls der erste Parameter ein Array ist
        if (is_array($backup_type)) {
            $args = $backup_type;
            $backup_type = isset($args['backup_type']) ? $args['backup_type'] : 'full';
            $schedule_num = isset($args['schedule_num']) ? $args['schedule_num'] : 2;
            $schedule_data = isset($args['schedule_data']) ? $args['schedule_data'] : [];
        }
        
        return $this->execute_backup($backup_type, 2, $schedule_data);
    }
    
    /**
     * Führt ein geplantes Backup aus
     */
    private function execute_backup($backup_type, $schedule_num, $schedule_data = []) {
        $logger = new Logger();
        $logger->log("=====================================");
        $logger->log("🚀 GEPLANTES BACKUP {$schedule_num} WIRD AUSGEFÜHRT");
        $logger->log("Typ: {$backup_type}");
        $logger->log("Zeitstempel: " . wp_date('Y-m-d H:i:s'));
        $logger->log("=====================================");
        
        try {
            // Neues 2.0 Backup-System verwenden
            $orchestrator = new \JenvaBackupMigration\Backup\BackupOrchestrator($backup_type, [
                'clean_database' => true,
                'exclude_cache' => true,
                'exclude_logs' => true,
                'exclude_other_backups' => true,
            ]);
            
            // Progress-Callback für Live-Updates
            $orchestrator->setProgressCallback(function($phase, $percent, $message) use ($logger) {
                $logger->log("📦 [{$phase}] {$percent}% - {$message}");
            });
            
            $result = $orchestrator->run();
            
            // Backup in DB speichern
            if ($result['success']) {
                global $wpdb;
                $table_name = $wpdb->prefix . 'jbm_backups';
                
                // Backup-Pfad: Die .jbm Datei
                $backup_file_path = $result['jbm_path'] ?? $result['path'];
                
                // Backup-Name aus dem Dateinamen extrahieren
                $backup_name = pathinfo($backup_file_path, PATHINFO_FILENAME);
                
                // Dateigröße ermitteln
                $file_size = file_exists($backup_file_path) ? filesize($backup_file_path) : ($result['size'] ?? 0);
                
                $wpdb->insert(
                    $table_name,
                    [
                        'backup_name' => $backup_name,
                        'backup_type' => $backup_type,
                        'backup_size' => $file_size,
                        'backup_date' => current_time('mysql'),
                        'backup_path' => $backup_file_path,
                        'backup_status' => 'completed',
                        'protected' => 0,
                    ],
                    ['%s', '%s', '%d', '%s', '%s', '%s', '%d']
                );
                
                $result['backup_name'] = $backup_name;
                $result['backup_size'] = $file_size;
                
                $logger->log("✅ Geplantes Backup {$schedule_num} ERFOLGREICH: " . $backup_name);
                $logger->log("Größe: " . size_format($result['backup_size']));
                
                // Letzte Ausführungszeit speichern
                update_option('jbm_schedule_' . $schedule_num . '_last_run', time());
            } else {
                $logger->log("❌ Geplantes Backup {$schedule_num} FEHLGESCHLAGEN: " . ($result['error'] ?? 'Unbekannter Fehler'), 'error');
            }
            
            // E-Mail-Benachrichtigung senden
            $settings = get_option('jbm_settings', []);
            
            if (!empty($settings['email_notifications']) && !empty($settings['email_address'])) {
                $this->send_notification_email($result, $settings['email_address'], $schedule_num);
            }
            
            $logger->log("=====================================");
            
            return $result;
            
        } catch (\Exception $e) {
            $logger->log("Kritischer Fehler bei geplant Backup {$schedule_num}: " . $e->getMessage(), 'error');
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Sendet eine E-Mail-Benachrichtigung
     */
    private function send_notification_email($result, $email, $schedule_num) {
        $subject = __('Backup-Benachrichtigung', 'jenva-backup-migration') . ' - ' . get_bloginfo('name');
        
        if ($result['success']) {
            $message = sprintf(__('Zeitplan %d: Das geplante Backup wurde erfolgreich erstellt.', 'jenva-backup-migration'), $schedule_num) . "\n\n";
            $message .= __('Backup-Name:', 'jenva-backup-migration') . ' ' . $result['backup_name'] . "\n";
            $message .= __('Größe:', 'jenva-backup-migration') . ' ' . (isset($result['backup_size']) ? size_format($result['backup_size']) : 'N/A') . "\n";
            $message .= __('Typ:', 'jenva-backup-migration') . ' ' . ($result['type'] ?? 'N/A') . "\n";
        } else {
            $message = sprintf(__('Zeitplan %d: Das geplante Backup ist fehlgeschlagen.', 'jenva-backup-migration'), $schedule_num) . "\n\n";
            $message .= __('Fehler:', 'jenva-backup-migration') . ' ' . ($result['message'] ?? 'Unbekannter Fehler') . "\n";
        }
        
        wp_mail($email, $subject, $message);
    }
    
    /**
     * Registriert benutzerdefinierte Zeitplan-Intervalle
     */
    public static function register_schedules($schedules) {
        $schedules['weekly'] = [
            'interval' => 604800, // 7 Tage
            'display' => __('Wöchentlich', 'jenva-backup-migration')
        ];
        
        $schedules['monthly'] = [
            'interval' => 2635200, // ~30.5 Tage
            'display' => __('Monatlich', 'jenva-backup-migration')
        ];
        
        return $schedules;
    }
}
