<?php
namespace JBM;

if (!defined('ABSPATH')) {
    exit;
}

class FileSystem {
    
    /**
     * Erlaubte Basisverzeichnisse für Dateioperationen
     * Sicherheit: Whitelist der erlaubten Verzeichnisse
     */
    private $allowed_base_dirs = [];
    
    public function __construct() {
        // Initialisiere erlaubte Verzeichnisse
        $this->allowed_base_dirs = [
            realpath(ABSPATH),
            realpath(WP_CONTENT_DIR),
            defined('JBM_BACKUP_DIR') ? realpath(JBM_BACKUP_DIR) : null,
            defined('JBM_TEMP_DIR') ? realpath(JBM_TEMP_DIR) : null,
        ];
        // Null-Werte entfernen (falls Verzeichnisse nicht existieren)
        $this->allowed_base_dirs = array_filter($this->allowed_base_dirs);
    }
    
    /**
     * Prüft ob ein Pfad innerhalb eines erlaubten Basisverzeichnisses liegt
     * Sicherheit: Verhindert Path Traversal Angriffe
     * 
     * @param string $path Der zu prüfende Pfad
     * @param string|null $required_base Optional: Spezifisches Basisverzeichnis das erfordert wird
     * @return bool True wenn Pfad sicher ist
     */
    private function is_path_safe($path, $required_base = null) {
        // Pfad auflösen (entfernt ../ etc.)
        $real_path = realpath($path);
        
        // Wenn Datei noch nicht existiert, Elternverzeichnis prüfen
        if ($real_path === false) {
            $parent = dirname($path);
            $real_path = realpath($parent);
            if ($real_path === false) {
                return false;
            }
            // Dateiname an aufgelösten Pfad anhängen
            $real_path = $real_path . '/' . basename($path);
        }
        
        // Normalisiere Pfadtrennzeichen
        $real_path = str_replace('\\', '/', $real_path);
        
        // Wenn spezifisches Basisverzeichnis erfordert
        if ($required_base !== null) {
            $real_base = realpath($required_base);
            if ($real_base === false) {
                return false;
            }
            $real_base = str_replace('\\', '/', $real_base);
            return strpos($real_path, $real_base) === 0;
        }
        
        // Prüfe gegen alle erlaubten Basisverzeichnisse
        foreach ($this->allowed_base_dirs as $base) {
            $safe_base = str_replace('\\', '/', $base);
            if (strpos($real_path, $safe_base) === 0) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Bereinigt einen Pfad von gefährlichen Zeichen
     * Sicherheit: Entfernt Path Traversal Versuche
     * 
     * @param string $path Der zu bereinigende Pfad
     * @return string Der bereinigte Pfad
     */
    private function sanitize_path($path) {
        // Entferne Null-Bytes (können zum Abschneiden von Strings missbraucht werden)
        $path = str_replace("\0", '', $path);
        
        // Normalisiere Pfadtrennzeichen
        $path = str_replace('\\', '/', $path);
        
        // Entferne doppelte Slashes
        $path = preg_replace('#/+#', '/', $path);
        
        // Entferne gefährliche Sequenzen am Anfang
        while (strpos($path, '../') === 0 || strpos($path, './') === 0) {
            $path = substr($path, strpos($path, '/') + 1);
        }
        
        return $path;
    }
    
    /**
     * Kopiert ein Verzeichnis rekursiv (verbesserte Version mit Path Traversal Schutz)
     * Sicherheit: Validiert alle Pfade gegen erlaubte Basisverzeichnisse
     */
    public function copy_directory($source, $destination, $exclude = [], $overwrite = false) {
        $files = [];
        
        if (!is_dir($source)) {
            error_log("JBM: Quellverzeichnis existiert nicht: $source");
            return $files;
        }
        
        // Sicherheit: Prüfe ob Quellverzeichnis erlaubt ist
        if (!$this->is_path_safe($source)) {
            error_log("JBM: Sicherheit - Quellverzeichnis nicht erlaubt: $source");
            return $files;
        }
        
        // Zielverzeichnis erstellen
        if (!is_dir($destination)) {
            if (!wp_mkdir_p($destination)) {
                error_log("JBM: Konnte Zielverzeichnis nicht erstellen: $destination");
                return $files;
            }
        }
        
        // Sicherheit: Prüfe ob Zielverzeichnis erlaubt ist
        if (!$this->is_path_safe($destination)) {
            error_log("JBM: Sicherheit - Zielverzeichnis nicht erlaubt: $destination");
            return $files;
        }
        
        // Reale Pfade für spätere Validierung
        $real_source = realpath($source);
        $real_destination = realpath($destination);
        
        if ($real_source === false || $real_destination === false) {
            error_log("JBM: Konnte Pfade nicht auflösen");
            return $files;
        }
        
        try {
            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($source, \RecursiveDirectoryIterator::SKIP_DOTS),
                \RecursiveIteratorIterator::SELF_FIRST
            );
            
            $file_count = 0;
            $error_count = 0;
            
            foreach ($iterator as $item) {
                try {
                    // Sicherheit: Verwende realpath für das aktuelle Item
                    $item_real_path = $item->getRealPath();
                    if ($item_real_path === false) {
                        // Symlink oder ungültiger Pfad - überspringen
                        continue;
                    }
                    
                    // Sicherheit: Prüfe dass Item innerhalb des Quellverzeichnisses liegt
                    if (strpos($item_real_path, $real_source) !== 0) {
                        error_log("JBM: Sicherheit - Path Traversal Versuch erkannt: $item_real_path");
                        continue;
                    }
                    
                    $relative_path = substr($item_real_path, strlen($real_source) + 1);
                    
                    // Sicherheit: Bereinige relativen Pfad
                    $relative_path = $this->sanitize_path($relative_path);
                    
                    // Ausgeschlossene Verzeichnisse überspringen
                    $skip = false;
                    foreach ($exclude as $excluded) {
                        $excluded = $this->sanitize_path($excluded);
                        if (strpos($relative_path, $excluded) === 0 || strpos($relative_path, $excluded . '/') !== false) {
                            $skip = true;
                            break;
                        }
                    }
                    
                    if ($skip) {
                        continue;
                    }
                    
                    $target = $real_destination . '/' . $relative_path;
                    
                    // Sicherheit: Finale Validierung des Zielpfads
                    // Verwende dirname für noch nicht existierende Dateien
                    $target_check_path = is_dir($target) ? $target : dirname($target);
                    if (!$this->is_path_safe($target_check_path, $real_destination)) {
                        error_log("JBM: Sicherheit - Ungültiger Zielpfad: $target");
                        continue;
                    }
                    
                    if ($item->isDir()) {
                        if (!is_dir($target)) {
                            if (!@wp_mkdir_p($target)) {
                                error_log("JBM: Konnte Verzeichnis nicht erstellen: $target");
                                $error_count++;
                            }
                        }
                    } else {
                        // Zielverzeichnis sicherstellen
                        $target_dir = dirname($target);
                        if (!is_dir($target_dir)) {
                            @wp_mkdir_p($target_dir);
                        }
                        
                        // Datei kopieren
                        if ($overwrite || !file_exists($target)) {
                            if (@copy($item_real_path, $target)) {
                                $files[] = $target;
                                $file_count++;
                            } else {
                                error_log("JBM: Konnte Datei nicht kopieren: " . $item_real_path . " -> $target");
                                $error_count++;
                            }
                        }
                    }
                    
                    // Fortschritt bei vielen Dateien
                    if ($file_count % 500 == 0 && $file_count > 0) {
                        error_log("JBM: Fortschritt: $file_count Dateien kopiert...");
                    }
                    
                } catch (\Exception $e) {
                    error_log("JBM: Fehler beim Kopieren von " . $item->getPathname() . ": " . $e->getMessage());
                    $error_count++;
                }
            }
            
            if ($error_count > 0) {
                error_log("JBM: Kopieren abgeschlossen mit $error_count Fehlern");
            }
            
        } catch (\Exception $e) {
            error_log("JBM: Kritischer Fehler beim Kopieren des Verzeichnisses: " . $e->getMessage());
        }
        
        return $files;
    }
    
    /**
     * Löscht ein Verzeichnis rekursiv
     * Sicherheit: Validiert Pfad gegen erlaubte Verzeichnisse
     */
    public function delete_directory($directory) {
        if (!is_dir($directory)) {
            return false;
        }
        
        // Sicherheit: Prüfe ob Verzeichnis erlaubt ist
        if (!$this->is_path_safe($directory)) {
            error_log("JBM: Sicherheit - Löschen von Verzeichnis nicht erlaubt: $directory");
            return false;
        }
        
        $real_directory = realpath($directory);
        if ($real_directory === false) {
            return false;
        }
        
        // Sicherheit: Kritische Verzeichnisse niemals löschen
        $protected_dirs = [
            realpath(ABSPATH),
            realpath(WP_CONTENT_DIR),
            realpath(WP_CONTENT_DIR . '/plugins'),
            realpath(WP_CONTENT_DIR . '/themes'),
            realpath(WP_CONTENT_DIR . '/uploads'),
        ];
        
        foreach ($protected_dirs as $protected) {
            if ($protected !== false && $real_directory === $protected) {
                error_log("JBM: Sicherheit - Kritisches Verzeichnis geschützt: $directory");
                return false;
            }
        }
        
        $files = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($directory, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::CHILD_FIRST
        );
        
        foreach ($files as $file) {
            $file_real_path = $file->getRealPath();
            
            // Sicherheit: Prüfe dass Datei innerhalb des zu löschenden Verzeichnisses liegt
            if ($file_real_path === false || strpos($file_real_path, $real_directory) !== 0) {
                error_log("JBM: Sicherheit - Path Traversal bei Löschung erkannt: " . $file->getPathname());
                continue;
            }
            
            if ($file->isDir()) {
                @rmdir($file_real_path);
            } else {
                @unlink($file_real_path);
            }
        }
        
        return @rmdir($directory);
    }
    
    /**
     * Berechnet die Größe eines Verzeichnisses
     * Sicherheit: Validiert Pfad gegen erlaubte Verzeichnisse
     */
    public function get_directory_size($directory) {
        $size = 0;
        
        if (!is_dir($directory)) {
            return $size;
        }
        
        // Sicherheit: Prüfe ob Verzeichnis erlaubt ist
        if (!$this->is_path_safe($directory)) {
            error_log("JBM: Sicherheit - Zugriff auf Verzeichnis nicht erlaubt: $directory");
            return $size;
        }
        
        $real_directory = realpath($directory);
        if ($real_directory === false) {
            return $size;
        }
        
        try {
            foreach (new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($directory, \RecursiveDirectoryIterator::SKIP_DOTS)) as $file) {
                // Sicherheit: Prüfe dass Datei innerhalb des Verzeichnisses liegt
                $file_real_path = $file->getRealPath();
                if ($file_real_path === false || strpos($file_real_path, $real_directory) !== 0) {
                    continue;
                }
                
                $size += $file->getSize();
            }
        } catch (\Exception $e) {
            error_log("JBM: Fehler bei Größenberechnung: " . $e->getMessage());
        }
        
        return $size;
    }
}

