<?php
/**
 * DatabaseImporter - Chunk-basierter Datenbank-Import
 * 
 * Importiert SQL-Dumps mit:
 * - Chunk-basierter Verarbeitung
 * - Automatischer Prefix-Ersetzung
 * - Fehlerbehandlung pro Statement
 * - Checkpoint-Unterstützung
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Restore;

use JenvaBackupMigration\Core\ChunkProcessor;

if (!defined('ABSPATH')) {
    exit;
}

class DatabaseImporter {
    
    /** @var int Statements pro Batch */
    const STATEMENTS_PER_BATCH = 100;
    
    /** @var array Statistiken */
    private $stats = [];
    
    /** @var callable|null Progress-Callback */
    private $progress_callback;
    
    /**
     * Setzt den Progress-Callback
     */
    public function setProgressCallback(callable $callback): void {
        $this->progress_callback = $callback;
    }
    
    /**
     * Importiert einen SQL-Dump
     * 
     * @param string $sql SQL-Inhalt
     * @param string|null $backup_prefix Prefix im Backup (null = aktuelles Prefix verwenden)
     * @return array Import-Ergebnis
     */
    public function import(string $sql, ?string $backup_prefix = null): array {
        global $wpdb;
        
        // Falls kein Prefix angegeben, verwende das aktuelle Prefix
        if ($backup_prefix === null) {
            $backup_prefix = $wpdb->prefix;
        }
        
        $this->stats = [
            'started_at' => current_time('mysql'),
            'statements_total' => 0,
            'statements_executed' => 0,
            'statements_failed' => 0,
            'tables_imported' => 0,
            'rows_affected' => 0,
            'errors' => [],
        ];
        
        // Prefix-Ersetzung
        $current_prefix = $wpdb->prefix;
        if ($backup_prefix !== $current_prefix) {
            $sql = $this->replacePrefix($sql, $backup_prefix, $current_prefix);
        }
        
        // SQL in Statements aufteilen
        $statements = $this->splitStatements($sql);
        $this->stats['statements_total'] = count($statements);
        
        // Fremdschlüssel-Checks deaktivieren
        $wpdb->query("SET FOREIGN_KEY_CHECKS = 0");
        
        // WICHTIG: Bestehende Tabellen löschen, die im Backup enthalten sind
        // Das ist das erwartete Verhalten bei einer Migration!
        $this->dropExistingTables($statements, $current_prefix);
        
        // Statements ausführen
        $tables_seen = [];
        
        foreach ($statements as $i => $statement) {
            $statement = trim($statement);
            
            if (empty($statement)) {
                continue;
            }
            
            // Fortschritt melden
            if ($this->progress_callback && $i % 50 === 0) {
                $percent = ($i / count($statements)) * 100;
                call_user_func($this->progress_callback, $percent, "Statement $i von " . count($statements));
            }
            
            // SICHERHEIT: Statement validieren bevor Ausführung
            if (!$this->isSafeStatement($statement)) {
                $this->stats['statements_failed']++;
                $this->stats['errors'][] = [
                    'statement' => substr($statement, 0, 200) . '...',
                    'error' => 'Unsafe SQL statement rejected by security validation',
                ];
                continue;
            }
            
            // Statement ausführen
            $result = $wpdb->query($statement);
            
            if ($result === false) {
                $this->stats['statements_failed']++;
                $this->stats['errors'][] = [
                    'statement' => substr($statement, 0, 200) . '...',
                    'error' => $wpdb->last_error,
                ];
            } else {
                $this->stats['statements_executed']++;
                
                if ($result > 0) {
                    $this->stats['rows_affected'] += $result;
                }
                
                // Tabelle tracken
                if (preg_match('/^(CREATE|INSERT INTO)\s+`?([^`\s]+)`?/i', $statement, $matches)) {
                    $tables_seen[$matches[2]] = true;
                }
            }
        }
        
        // Fremdschlüssel-Checks wieder aktivieren
        $wpdb->query("SET FOREIGN_KEY_CHECKS = 1");
        
        $this->stats['tables_imported'] = count($tables_seen);
        $this->stats['completed_at'] = current_time('mysql');
        
        return $this->stats;
    }
    
    /**
     * Löscht alle bestehenden Tabellen, die im Backup enthalten sind
     * 
     * WICHTIG: Dies ist das erwartete Verhalten bei einer Migration!
     * Ohne dies werden Tabellen/Daten nur hinzugefügt statt ersetzt.
     * 
     * @param array $statements SQL-Statements
     * @param string $current_prefix Aktueller Tabellen-Prefix
     */
    private function dropExistingTables(array $statements, string $current_prefix): void {
        global $wpdb;
        
        $tables_to_drop = [];
        
        // Finde alle CREATE TABLE Statements und extrahiere Tabellennamen
        foreach ($statements as $statement) {
            if (preg_match('/^CREATE TABLE\s+(?:IF NOT EXISTS\s+)?`?([^`\s]+)`?/i', $statement, $matches)) {
                $tables_to_drop[] = $matches[1];
            }
        }
        
        
        // Tabellen in umgekehrter Reihenfolge löschen (wegen Fremdschlüsseln)
        $tables_to_drop = array_reverse($tables_to_drop);
        
        foreach ($tables_to_drop as $table) {
            // SICHERHEIT: Tabellennamen validieren
            if (!$this->isValidTableName($table)) {
                $this->stats['errors'][] = [
                    'statement' => "DROP TABLE IF EXISTS `{$table}`",
                    'error' => 'Invalid table name detected - rejected by security validation',
                ];
                continue;
            }
            
            // Sicherheit: Nur Tabellen mit korrektem Prefix löschen
            if (strpos($table, $current_prefix) !== 0) {
                $this->stats['errors'][] = [
                    'statement' => "DROP TABLE IF EXISTS `{$table}`",
                    'error' => 'Table prefix mismatch - rejected by security validation',
                ];
                continue;
            }
            
            // Tabellennamen sicher escapen
            $safe_table = $this->escapeTableName($table);
            if ($safe_table === false) {
                $this->stats['errors'][] = [
                    'statement' => "DROP TABLE IF EXISTS `{$table}`",
                    'error' => 'Failed to escape table name',
                ];
                continue;
            }
            
            // Sicher ausführen
            $result = $wpdb->query("DROP TABLE IF EXISTS {$safe_table}");
            if ($result === false) {
                $this->stats['errors'][] = [
                    'statement' => "DROP TABLE IF EXISTS {$safe_table}",
                    'error' => $wpdb->last_error,
                ];
            }
        }
    }
    
    /**
     * Ersetzt den Tabellen-Prefix
     * 
     * @param string $sql SQL-Inhalt
     * @param string $old_prefix Alter Prefix
     * @param string $new_prefix Neuer Prefix
     * @return string
     */
    private function replacePrefix(string $sql, string $old_prefix, string $new_prefix): string {
        // Tabellennamen ersetzen
        $patterns = [
            // CREATE TABLE `prefix_xxx`
            "/CREATE TABLE (IF NOT EXISTS )?`{$old_prefix}/i" => "CREATE TABLE $1`{$new_prefix}",
            // DROP TABLE IF EXISTS `prefix_xxx`
            "/DROP TABLE IF EXISTS `{$old_prefix}/i" => "DROP TABLE IF EXISTS `{$new_prefix}",
            // INSERT INTO `prefix_xxx`
            "/INSERT INTO `{$old_prefix}/i" => "INSERT INTO `{$new_prefix}",
            // UPDATE `prefix_xxx`
            "/UPDATE `{$old_prefix}/i" => "UPDATE `{$new_prefix}",
            // ALTER TABLE `prefix_xxx`
            "/ALTER TABLE `{$old_prefix}/i" => "ALTER TABLE `{$new_prefix}",
            // REFERENCES `prefix_xxx`
            "/REFERENCES `{$old_prefix}/i" => "REFERENCES `{$new_prefix}",
        ];
        
        foreach ($patterns as $pattern => $replacement) {
            $sql = preg_replace($pattern, $replacement, $sql);
        }
        
        // Prefix in Daten ersetzen (für wp_options, wp_usermeta etc.)
        $data_patterns = [
            // option_name Werte
            "/('{$old_prefix})([^']+')/" => "'{$new_prefix}$2",
            // meta_key Werte in usermeta
            "/({$old_prefix})(capabilities|user_level|dashboard_quick_press_last_post_id)/" => "{$new_prefix}$2",
        ];
        
        foreach ($data_patterns as $pattern => $replacement) {
            $sql = preg_replace($pattern, $replacement, $sql);
        }
        
        return $sql;
    }
    
    /**
     * Teilt SQL in einzelne Statements
     * 
     * @param string $sql SQL-Inhalt
     * @return array Statements
     */
    private function splitStatements(string $sql): array {
        // Einfacher Split an Semikolon (für komplexere Fälle Verbesserung nötig)
        $statements = [];
        $current = '';
        $in_string = false;
        $string_char = '';
        
        $length = strlen($sql);
        
        for ($i = 0; $i < $length; $i++) {
            $char = $sql[$i];
            $prev = $i > 0 ? $sql[$i - 1] : '';
            
            // String-Tracking
            if (($char === "'" || $char === '"') && $prev !== '\\') {
                if (!$in_string) {
                    $in_string = true;
                    $string_char = $char;
                } elseif ($char === $string_char) {
                    $in_string = false;
                }
            }
            
            // Semikolon außerhalb von Strings
            if ($char === ';' && !$in_string) {
                $statement = trim($current);
                if (!empty($statement) && !$this->isComment($statement)) {
                    $statements[] = $statement;
                }
                $current = '';
            } else {
                $current .= $char;
            }
        }
        
        // Letztes Statement
        $statement = trim($current);
        if (!empty($statement) && !$this->isComment($statement)) {
            $statements[] = $statement;
        }
        
        return $statements;
    }
    
    /**
     * Prüft ob ein Statement nur ein Kommentar ist
     */
    private function isComment(string $statement): bool {
        $statement = trim($statement);
        return strpos($statement, '--') === 0 || 
               strpos($statement, '#') === 0 ||
               (strpos($statement, '/*') === 0 && strpos($statement, '*/') !== false && 
                strpos($statement, '/*!') !== 0);
    }
    
    /**
     * Importiert nur bestimmte Tabellen
     * 
     * @param string $sql SQL-Inhalt
     * @param array $tables Tabellen die importiert werden sollen
     * @param string|null $backup_prefix Prefix im Backup (null = aktuelles Prefix verwenden)
     * @return array
     */
    public function importTables(string $sql, array $tables, ?string $backup_prefix = null): array {
        global $wpdb;
        
        // Falls kein Prefix angegeben, verwende das aktuelle Prefix
        if ($backup_prefix === null) {
            $backup_prefix = $wpdb->prefix;
        }
        
        // Prefix anpassen für Suche
        $current_prefix = $wpdb->prefix;
        
        // Statements extrahieren und filtern
        $all_statements = $this->splitStatements($sql);
        $filtered_statements = [];
        
        foreach ($all_statements as $statement) {
            foreach ($tables as $table) {
                // Tabelle mit altem und neuem Prefix prüfen
                $old_table = $backup_prefix . $table;
                $new_table = $current_prefix . $table;
                
                if (stripos($statement, $old_table) !== false || 
                    stripos($statement, $new_table) !== false) {
                    $filtered_statements[] = $statement;
                    break;
                }
            }
        }
        
        // Prefix ersetzen in gefilterten Statements
        $sql_filtered = implode(";\n", $filtered_statements);
        
        if ($backup_prefix !== $current_prefix) {
            $sql_filtered = $this->replacePrefix($sql_filtered, $backup_prefix, $current_prefix);
        }
        
        return $this->import($sql_filtered, $current_prefix);
    }
    
    /**
     * Validiert ob ein SQL-Statement sicher ausgeführt werden darf
     * 
     * Whitelist-Ansatz: Nur erlaubte SQL-Operationen werden zugelassen
     * 
     * @param string $statement SQL-Statement
     * @return bool True wenn sicher, sonst False
     */
    private function isSafeStatement(string $statement): bool {
        $statement = trim($statement);
        
        // Erlaubte SQL-Operationen (Whitelist)
        $allowed_operations = [
            'CREATE TABLE',
            'INSERT INTO',
            'UPDATE',
            'DELETE FROM',
            'ALTER TABLE',
            'DROP TABLE',
            'SET',
            'LOCK TABLES',
            'UNLOCK TABLES',
        ];
        
        // Prüfe ob Statement mit erlaubter Operation beginnt
        $is_allowed = false;
        foreach ($allowed_operations as $operation) {
            if (stripos($statement, $operation) === 0) {
                $is_allowed = true;
                break;
            }
        }
        
        if (!$is_allowed) {
            return false;
        }
        
        // Zusätzliche Sicherheitsprüfungen für bestimmte Operationen
        $upper_statement = strtoupper($statement);
        
        // Verhindere gefährliche Operationen
        $dangerous_patterns = [
            '/\bDROP\s+DATABASE\b/i',
            '/\bCREATE\s+DATABASE\b/i',
            '/\bUSE\s+[^;]+/i',
            '/\bGRANT\b/i',
            '/\bREVOKE\b/i',
            '/\bFLUSH\s+PRIVILEGES\b/i',
            '/\bLOAD\s+FILE\b/i',
            '/\bINTO\s+OUTFILE\b/i',
            '/\bINTO\s+DUMPFILE\b/i',
            '/\bEXEC\b/i',
            '/\bEXECUTE\b/i',
            '/\bCALL\b/i',
            '/\bTRUNCATE\s+TABLE\b/i', // Gefährlich, nur erlauben wenn explizit gewünscht
        ];
        
        foreach ($dangerous_patterns as $pattern) {
            if (preg_match($pattern, $statement)) {
                return false;
            }
        }
        
        // Validiere Tabellennamen in CREATE/DROP/INSERT/UPDATE/DELETE
        if (preg_match('/^(CREATE|DROP|INSERT INTO|UPDATE|DELETE FROM|ALTER TABLE)\s+(?:IF NOT EXISTS\s+)?(?:IF EXISTS\s+)?`?([^`\s]+)`?/i', $statement, $matches)) {
            $table_name = isset($matches[2]) ? $matches[2] : '';
            if (!empty($table_name) && !$this->isValidTableName($table_name)) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Validiert einen Tabellennamen auf sichere Zeichen
     * 
     * @param string $table_name Tabellenname
     * @return bool True wenn gültig
     */
    private function isValidTableName(string $table_name): bool {
        // Nur alphanumerische Zeichen, Unterstriche und Bindestriche erlaubt
        // Muss mit Buchstabe oder Zahl beginnen
        // Maximale Länge: 64 Zeichen (MySQL-Limit)
        if (strlen($table_name) > 64) {
            return false;
        }
        
        // Nur erlaubte Zeichen
        return preg_match('/^[a-zA-Z0-9_][a-zA-Z0-9_-]*$/', $table_name) === 1;
    }
    
    /**
     * Escaped einen Tabellennamen sicher für SQL-Queries
     * 
     * @param string $table_name Tabellenname
     * @return string|false Escaped Name oder false bei ungültigem Namen
     */
    private function escapeTableName(string $table_name) {
        global $wpdb;
        
        if (!$this->isValidTableName($table_name)) {
            return false;
        }
        
        // WordPress' $wpdb->_escape() für Backticks verwenden
        return '`' . str_replace('`', '``', $table_name) . '`';
    }
    
    /**
     * Getter für Statistiken
     */
    public function getStats(): array {
        return $this->stats;
    }
}

