<?php
/**
 * RestoreWizard - Geführter Restore-Prozess
 * 
 * Der Wizard führt durch den Restore-Prozess:
 * 1. Backup validieren
 * 2. Datenbank wiederherstellen
 * 3. Dateien wiederherstellen
 * 4. Plugins installieren/reaktivieren
 * 5. URL-Migration durchführen
 * 6. Caches leeren
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Restore;

use JenvaBackupMigration\Core\Container;
use JenvaBackupMigration\Core\ChunkProcessor;
use JenvaBackupMigration\Compatibility\BricksHandler;
use JenvaBackupMigration\Compatibility\BuilderHandler;

if (!defined('ABSPATH')) {
    exit;
}

class RestoreWizard {
    
    /** @var Container */
    private $container;
    
    /** @var DatabaseImporter */
    private $db_importer;
    
    /** @var PluginReconstructor */
    private $plugin_reconstructor;
    
    /** @var FileRestorer */
    private $file_restorer;
    
    /** @var UrlMigrator */
    private $url_migrator;
    
    /** @var BricksHandler */
    private $bricks_handler;
    
    /** @var BuilderHandler */
    private $builder_handler;
    
    /** @var ChunkProcessor */
    private $chunk_processor;
    
    /** @var callable|null Progress-Callback */
    private $progress_callback;
    
    /** @var string Original Site-URL (vor Restore) */
    private $original_site_url;
    
    /** @var array Restore-Statistiken */
    private $stats = [
        'phase' => '',
        'current_step' => 0,
        'total_steps' => 6,
        'errors' => [],
        'warnings' => [],
    ];
    
    /** @var array Konfiguration */
    private $config;
    
    /**
     * Erstellt einen neuen RestoreWizard
     * 
     * @param string $backup_path Pfad zum Backup (.jbm oder Verzeichnis)
     * @param array $config Konfiguration
     */
    public function __construct(string $backup_path, array $config = []) {
        // KRITISCH: Aktuelle URL VORHER speichern!
        $this->original_site_url = get_site_url();
        
        $this->config = array_merge([
            'restore_database' => true,
            'restore_files' => true,
            'restore_plugins' => true,
            'migrate_urls' => true,
            'clear_caches' => true,
            'overwrite_wp_config' => false,
            'keep_existing_plugins' => false, // Default: Plugins entfernen, die nicht im Backup sind
            'temp_dir' => WP_CONTENT_DIR . '/jenva-backup-migration/temp/',
        ], $config);
        
        // Container öffnen
        $this->container = Container::open($backup_path);
        
        // Komponenten initialisieren
        $this->initializeComponents();
    }
    
    /**
     * Setzt den Progress-Callback
     * 
     * @param callable $callback function($phase, $percent, $message)
     */
    public function setProgressCallback(callable $callback): void {
        $this->progress_callback = $callback;
    }
    
    /**
     * Führt den kompletten Restore durch
     * 
     * @return array Restore-Ergebnis
     */
    public function run(): array {
        try {
            
            $this->reportProgress('start', 0, 'Restore wird gestartet...');
            
            // Step 1: Validierung
            $this->step1_validate();
            
            // Step 2: Datenbank
            if ($this->config['restore_database']) {
                $this->step2_database();
            }
            
            // Step 3: Dateien
            if ($this->config['restore_files']) {
                $this->step3_files();
            }
            
            // Step 4: Plugins
            if ($this->config['restore_plugins']) {
                $this->step4_plugins();
            }
            
            // Step 5: URL-Migration
            if ($this->config['migrate_urls']) {
                $this->step5_urls();
            }
            
            // Step 6: Finalisierung
            $this->step6_finalize();
            
            $this->reportProgress('complete', 100, 'Restore abgeschlossen!');
            
            
            
            
            return [
                'success' => true,
                'stats' => $this->stats,
                'new_url' => $this->original_site_url,
                'old_url' => $this->container->getManifest()->getSiteUrl(),
            ];
            
        } catch (\Exception $e) {
            
            $this->stats['errors'][] = $e->getMessage();
            
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'stats' => $this->stats,
            ];
        }
    }
    
    /**
     * Step 1: Backup validieren
     */
    private function step1_validate(): void {
        $this->stats['phase'] = 'validate';
        $this->stats['current_step'] = 1;
        $this->reportProgress('validate', 5, 'Validiere Backup...');
        
        // Integrität prüfen
        $verification = $this->container->verify();
        
        if (!$verification['valid']) {
            throw new \Exception('Backup-Integrität konnte nicht verifiziert werden: ' . implode(', ', $verification['errors']));
        }
        
        $this->stats['validation'] = $verification;
        $this->reportProgress('validate', 10, 'Backup validiert');
    }
    
    /**
     * Step 2: Datenbank wiederherstellen
     */
    private function step2_database(): void {
        $this->stats['phase'] = 'database';
        $this->stats['current_step'] = 2;
        $this->reportProgress('database', 15, 'Stelle Datenbank wieder her...');
        
        // SQL aus Container holen
        $sql = $this->container->getDatabase();
        
        if (empty($sql)) {
            $this->stats['warnings'][] = 'Keine Datenbank im Backup gefunden';
            return;
        }
        
        // Prefix aus Manifest
        $backup_prefix = $this->container->getManifest()->getDbPrefix();
        
        // Datenbank importieren
        $result = $this->db_importer->import($sql, $backup_prefix);
        
        // KRITISCH: URLs SOFORT nach DB-Import setzen, BEVOR Local's URL-Fixer reagiert!
        $this->setUrlsImmediately();
        
        $this->stats['database'] = $result;
        $this->reportProgress('database', 40, "Datenbank wiederhergestellt: {$result['tables_imported']} Tabellen");
    }
    
    /**
     * Setzt die URLs SOFORT nach dem DB-Import
     * 
     * WICHTIG: Local by Flywheel hat einen URL-Fixer, der die Datenbank überwacht.
     * Wenn wir die URLs nicht SOFORT setzen, läuft Local's URL-Fixer und zerstört
     * die serialisierten Daten (er versteht PHP-Serialisierung nicht).
     */
    private function setUrlsImmediately(): void {
        global $wpdb;
        
        $new_url = $this->original_site_url;
        
        // Direkte SQL-Queries - KEIN WordPress-Cache, KEIN Delay
        $wpdb->query($wpdb->prepare(
            "UPDATE {$wpdb->options} SET option_value = %s WHERE option_name = 'siteurl'",
            $new_url
        ));
        
        $wpdb->query($wpdb->prepare(
            "UPDATE {$wpdb->options} SET option_value = %s WHERE option_name = 'home'",
            $new_url
        ));
        
    }
    
    /**
     * Step 3: Dateien wiederherstellen
     */
    private function step3_files(): void {
        $this->stats['phase'] = 'files';
        $this->stats['current_step'] = 3;
        $this->reportProgress('files', 45, 'Stelle Dateien wieder her...');
        
        // Uploads wiederherstellen
        $uploads_result = $this->file_restorer->restoreUploads($this->container);
        
        // Themes wiederherstellen
        $themes_result = $this->file_restorer->restoreThemes($this->container);
        
        $this->stats['files'] = [
            'uploads' => $uploads_result,
            'themes' => $themes_result,
        ];
        
        $total_files = ($uploads_result['files_restored'] ?? 0) + ($themes_result['files_restored'] ?? 0);
        $this->reportProgress('files', 65, "Dateien wiederhergestellt: $total_files");
    }
    
    /**
     * Step 4: Plugins installieren
     */
    private function step4_plugins(): void {
        $this->stats['phase'] = 'plugins';
        $this->stats['current_step'] = 4;
        $this->reportProgress('plugins', 70, 'Installiere Plugins...');
        
        // Plugin-Inventar aus Manifest
        $plugins = $this->container->getManifest()->getPlugins();
        
        if (empty($plugins)) {
            $this->stats['warnings'][] = 'Kein Plugin-Inventar im Backup';
            return;
        }
        
        // Plugins rekonstruieren
        $keep_existing = $this->config['keep_existing_plugins'] ?? false;
        $result = $this->plugin_reconstructor->reconstruct($plugins, $this->container, $keep_existing);
        
        $this->stats['plugins'] = $result;
        $this->reportProgress('plugins', 80, "Plugins installiert: {$result['installed']} von {$result['total']}");
        
        if (!$keep_existing && isset($result['removed'])) {
            $this->reportProgress('plugins', 82, "Plugins entfernt: {$result['removed']}");
        }
    }
    
    /**
     * Step 5: URL-Migration
     */
    private function step5_urls(): void {
        $this->stats['phase'] = 'urls';
        $this->stats['current_step'] = 5;
        $this->reportProgress('urls', 82, 'Migriere URLs...');
        
        $backup_url = $this->container->getManifest()->getSiteUrl();
        $current_url = $this->original_site_url; // Die URL VOR dem Restore!
        
        if ($backup_url === $current_url) {
            $this->stats['urls'] = ['skipped' => true, 'reason' => 'URLs identisch'];
            return;
        }
        
        // URL-Migration durchführen (generisch für alle Tabellen)
        $result = $this->url_migrator->migrate($backup_url, $current_url);
        
        // Builder-spezifische URL-Migration (für alle Builder-Plugins)
        if ($this->builder_handler->hasBuilderData()) {
            $this->reportProgress('urls', 85, 'Migriere Builder-URLs...');
            $builder_result = $this->builder_handler->migrateUrls($backup_url, $current_url);
            $result['builders'] = $builder_result;
        }
        
        // Bricks-spezifische URL-Migration (für zusätzliche Bricks-spezifische Daten)
        if ($this->bricks_handler->isBricksActive()) {
            $bricks_result = $this->bricks_handler->migrateUrls($backup_url, $current_url);
            $result['bricks'] = $bricks_result;
        }
        
        $this->stats['urls'] = $result;
        $this->reportProgress('urls', 92, "URLs migriert: {$result['changes']} Änderungen");
    }
    
    /**
     * Step 6: Finalisierung
     */
    private function step6_finalize(): void {
        $this->stats['phase'] = 'finalize';
        $this->stats['current_step'] = 6;
        $this->reportProgress('finalize', 94, 'Finalisiere Restore...');
        
        
        
        // Builder-Daten reparieren (für alle Builder-Plugins)
        if ($this->builder_handler->hasBuilderData()) {
            $this->reportProgress('finalize', 95, 'Repariere Builder-Daten...');
            $this->builder_handler->repairAll();
        }
        
        // Bricks-Daten reparieren (zusätzlich, falls aktiv)
        if ($this->bricks_handler->isBricksActive()) {
            $this->reportProgress('finalize', 96, 'Repariere Bricks-Daten...');
            $this->bricks_handler->repairAll();
            
        }
        
        // Caches leeren
        if ($this->config['clear_caches']) {
            $this->reportProgress('finalize', 97, 'Leere Caches...');
            $this->clearAllCaches();
        }
        
        // KRITISCH: Permalink-Struktur reparieren (falls durch $wpdb->prepare() korrupt)
        $this->repairPermalinkStructure();
        
        // Permalinks flush
        flush_rewrite_rules();
        
        
        // KRITISCH: Backup der Bricks-Options erstellen (für Local's URL-Fixer Workaround)
        $this->backupBricksOptions();
        
        // KRITISCH: MU-Plugin für Reparatur erstellen
        $this->createFrontendDebugPlugin();
        
        $this->reportProgress('finalize', 99, 'Restore finalisiert');
    }
    
    /**
     * Repariert die Permalink-Struktur, falls sie durch $wpdb->prepare() korrupt wurde
     * 
     * WordPress' $wpdb->prepare() ersetzt % Zeichen durch einen SHA256-Hash-Platzhalter.
     * Das passiert wenn der Wert selbst % enthält (wie %postname%, %category%).
     * Diese Funktion stellt die korrekten Platzhalter wieder her.
     */
    private function repairPermalinkStructure(): void {
        global $wpdb;
        
        // Der Hash, den WordPress als Platzhalter verwendet
        $wpdb_placeholder_hash = '6324c777ee4eb15d8940d6e9863372be13010fda00e44b13fb21227e94493171';
        
        $permalink_structure = $wpdb->get_var(
            "SELECT option_value FROM {$wpdb->options} WHERE option_name = 'permalink_structure'"
        );
        
        // Prüfe ob der Hash vorhanden ist
        if (strpos($permalink_structure, $wpdb_placeholder_hash) !== false) {
            // Ersetze {hash} und {hash} zurück zu %
            $fixed_structure = str_replace(
                ['{' . $wpdb_placeholder_hash . '}', $wpdb_placeholder_hash],
                '%',
                $permalink_structure
            );
            
            // Direkte SQL-Query um WordPress-Cache zu umgehen
            $wpdb->query($wpdb->prepare(
                "UPDATE {$wpdb->options} SET option_value = %s WHERE option_name = 'permalink_structure'",
                $fixed_structure
            ));
            
        }
        
        // Prüfe auch rewrite_rules
        $rewrite_rules = $wpdb->get_var(
            "SELECT option_value FROM {$wpdb->options} WHERE option_name = 'rewrite_rules'"
        );
        
        if ($rewrite_rules && strpos($rewrite_rules, $wpdb_placeholder_hash) !== false) {
            // Rewrite-Regeln komplett löschen - werden durch flush_rewrite_rules() neu generiert
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name = 'rewrite_rules'");
            
        }
        
        // Cache leeren
        wp_cache_delete('permalink_structure', 'options');
        wp_cache_delete('rewrite_rules', 'options');
        wp_cache_delete('alloptions', 'options');
    }
    
    /**
     * Speichert ein Backup der korrekten Bricks-Options
     * 
     * Local by Flywheel hat einen automatischen URL-Fixer, der die Datenbank
     * überwacht und URLs ersetzt. Dieser Fixer versteht PHP-Serialisierung NICHT
     * und zerstört die serialisierten Daten. Wir speichern die korrekten Werte,
     * damit das MU-Plugin sie beim nächsten Request wiederherstellen kann.
     */
    private function backupBricksOptions(): void {
        global $wpdb;
        
        $backup_dir = WP_CONTENT_DIR . '/jenva-backup-migration/';
        if (!is_dir($backup_dir)) {
            wp_mkdir_p($backup_dir);
        }
        
        $backup_file = $backup_dir . '.bricks_options_backup.json';
        
        $options_to_backup = [
            'bricks_color_palette',
            'bricks_global_classes',
            'bricks_global_settings',
            'bricks_global_elements',
            'bricks_breakpoints',
            'bricks_remote_templates',
            'bricks_font_face_rules',
            // Kritische WordPress-Optionen
            'siteurl',
            'home',
            'permalink_structure',
        ];
        
        $backup_data = [
            'timestamp' => time(),
            'new_url' => $this->original_site_url,
            'old_url' => $this->container->getManifest()->getSiteUrl(), // KRITISCH: Alte URL für spätere Migration
            'options' => [],
            'postmeta' => [], // KRITISCH: Auch Bricks-Postmeta sichern
        ];
        
        foreach ($options_to_backup as $opt_name) {
            $value = $wpdb->get_var($wpdb->prepare(
                "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s",
                $opt_name
            ));
            if ($value !== null) {
                $backup_data['options'][$opt_name] = $value;
            }
        }
        
        // KRITISCH: Bricks-Postmeta sichern (für Wiederherstellung nach Local's URL-Fixer)
        $bricks_meta_keys = ['_bricks_page_content_2', '_bricks_page_header_2', '_bricks_page_footer_2', '_bricks_template_settings'];
        foreach ($bricks_meta_keys as $meta_key) {
            $entries = $wpdb->get_results($wpdb->prepare(
                "SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE meta_key = %s",
                $meta_key
            ), ARRAY_A);
            
            foreach ($entries as $entry) {
                if (!empty($entry['meta_value']) && strlen($entry['meta_value']) > 10) {
                    $backup_data['postmeta'][] = [
                        'post_id' => $entry['post_id'],
                        'meta_key' => $meta_key,
                        'meta_value' => $entry['meta_value'],
                    ];
                }
            }
        }
        
        file_put_contents($backup_file, json_encode($backup_data));
        
    }
    
    /**
     * Erstellt ein MU-Plugin für Post-Restore Reparatur
     * 
     * Dieses Plugin prüft beim nächsten Request, ob Local by Flywheel's
     * URL-Fixer die Bricks-Options zerstört hat, und stellt sie aus dem
     * Backup wieder her.
     */
    private function createFrontendDebugPlugin(): void {
        $mu_plugins_dir = WPMU_PLUGIN_DIR;
        if (!is_dir($mu_plugins_dir)) {
            wp_mkdir_p($mu_plugins_dir);
        }
        
        $debug_file = $mu_plugins_dir . '/jenva-restore-debug.php';
        $log_path = '/home/desatado85/Local Sites/the-new-backup-test/app/public/wp-content/plugins/jenva-backup-migration/.cursor/debug.log';
        
        $php_code = <<<'PHP'
<?php
/**
 * KRITISCHES MU-Plugin: Repariert Daten nach Local's URL-Fixer
 * 
 * Local by Flywheel hat einen automatischen URL-Fixer, der die Datenbank
 * überwacht und URLs ersetzt. Dieser Fixer versteht PHP-Serialisierung NICHT
 * und zerstört die serialisierten Daten.
 * 
 * Dieses Plugin stellt die korrekten Werte aus einem Backup wieder her.
 * Es wird bei JEDEM Request ausgeführt, bis das Backup verarbeitet wurde.
 */

// SOFORT beim Start: Backup verarbeiten
add_action('muplugins_loaded', function() {
    global $wpdb;
    $backup_file = WP_CONTENT_DIR . '/jenva-backup-migration/.bricks_options_backup.json';
    $log_path = '{{LOG_PATH}}';
    
    if (!file_exists($backup_file)) {
        // Kein Backup - MU-Plugin löschen
        @unlink(__FILE__);
        return;
    }
    
    $backup_data = json_decode(file_get_contents($backup_file), true);
    if (!$backup_data || !isset($backup_data['options'])) {
        @unlink($backup_file);
        @unlink(__FILE__);
        return;
    }
    
    $repaired_options = [];
    $repaired_postmeta = 0;
    $new_url = $backup_data['new_url'] ?? '';
    $old_url = $backup_data['old_url'] ?? '';
    
    // KRITISCH: IMMER siteurl und home zuerst setzen (um Redirects zu verhindern)
    if (!empty($new_url)) {
        $current_siteurl = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = 'siteurl'");
        $current_home = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = 'home'");
        
        // Nur reparieren wenn falsch
        if ($current_siteurl !== $new_url || $current_home !== $new_url) {
            $wpdb->query($wpdb->prepare(
                "UPDATE {$wpdb->options} SET option_value = %s WHERE option_name = 'siteurl'",
                $new_url
            ));
            $wpdb->query($wpdb->prepare(
                "UPDATE {$wpdb->options} SET option_value = %s WHERE option_name = 'home'",
                $new_url
            ));
            
            @file_put_contents($log_path, json_encode([
                'location' => 'mu-plugin:url_fix',
                'message' => 'URLs korrigiert!',
                'data' => [
                    'old_siteurl' => $current_siteurl,
                    'old_home' => $current_home,
                    'new_url' => $new_url,
                ],
                'timestamp' => round(microtime(true)*1000),
                'sessionId' => 'debug-session',
                'hypothesisId' => 'AC'
            ])."\n", FILE_APPEND);
        }
    }
    
    // Bricks-Optionen aus Backup wiederherstellen
    foreach ($backup_data['options'] as $opt_name => $opt_value) {
        // Überspringe URL-Optionen (bereits oben behandelt)
        if (in_array($opt_name, ['siteurl', 'home'])) continue;
        
        $result = $wpdb->update(
            $wpdb->options,
            ['option_value' => $opt_value],
            ['option_name' => $opt_name],
            ['%s'],
            ['%s']
        );
        $repaired_options[$opt_name] = $result !== false;
    }
    
    @file_put_contents($log_path, json_encode([
        'location' => 'mu-plugin:bricks_options_repair',
        'message' => 'Bricks-Options aus Backup wiederhergestellt!',
        'data' => ['repaired_options' => $repaired_options],
        'timestamp' => round(microtime(true)*1000),
        'sessionId' => 'debug-session',
        'hypothesisId' => 'Y'
    ])."\n", FILE_APPEND);
    
    // KRITISCH: Bricks-Postmeta aus Backup wiederherstellen
    if (!empty($backup_data['postmeta'])) {
        foreach ($backup_data['postmeta'] as $entry) {
            $result = $wpdb->update(
                $wpdb->postmeta,
                ['meta_value' => $entry['meta_value']],
                [
                    'post_id' => $entry['post_id'],
                    'meta_key' => $entry['meta_key'],
                ],
                ['%s'],
                ['%d', '%s']
            );
            if ($result !== false) {
                $repaired_postmeta++;
            }
        }
        
        @file_put_contents($log_path, json_encode([
            'location' => 'mu-plugin:bricks_postmeta_repair',
            'message' => 'Bricks-Postmeta aus Backup wiederhergestellt!',
            'data' => [
                'total_entries' => count($backup_data['postmeta']),
                'repaired' => $repaired_postmeta,
            ],
            'timestamp' => round(microtime(true)*1000),
            'sessionId' => 'debug-session',
            'hypothesisId' => 'AJ'
        ])."\n", FILE_APPEND);
    }
    
    // Permalink-Struktur reparieren (falls Hash vorhanden)
    $permalink = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = 'permalink_structure'");
    $hash = '6324c777ee4eb15d8940d6e9863372be13010fda00e44b13fb21227e94493171';
    if ($permalink && strpos($permalink, $hash) !== false) {
        $fixed = str_replace(['{' . $hash . '}', $hash], '%', $permalink);
        $wpdb->query($wpdb->prepare(
            "UPDATE {$wpdb->options} SET option_value = %s WHERE option_name = 'permalink_structure'",
            $fixed
        ));
        
        // Rewrite-Regeln löschen (werden neu generiert)
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name = 'rewrite_rules'");
        
        @file_put_contents($log_path, json_encode([
            'location' => 'mu-plugin:permalink_fix',
            'message' => 'Permalink-Struktur repariert!',
            'data' => ['old' => $permalink, 'fixed' => $fixed],
            'timestamp' => round(microtime(true)*1000),
            'sessionId' => 'debug-session',
            'hypothesisId' => 'AA'
        ])."\n", FILE_APPEND);
    }
    
    // Aufräumen
    @unlink($backup_file);
    @unlink(__FILE__);
    
    // Cache leeren
    wp_cache_flush();
    
    @file_put_contents($log_path, json_encode([
        'location' => 'mu-plugin:complete',
        'message' => 'MU-Plugin Reparatur abgeschlossen',
        'data' => [
            'options_repaired' => count($repaired_options),
            'postmeta_repaired' => $repaired_postmeta,
        ],
        'timestamp' => round(microtime(true)*1000),
        'sessionId' => 'debug-session',
        'hypothesisId' => 'Y'
    ])."\n", FILE_APPEND);
}, 0); // Priorität 0 = so früh wie möglich
PHP;
        
        // Platzhalter ersetzen
        $php_code = str_replace('{{LOG_PATH}}', $log_path, $php_code);
        
        file_put_contents($debug_file, $php_code);
        
    }
    
    // ========================================
    // HELPER METHODEN
    // ========================================
    
    /**
     * Initialisiert alle Komponenten
     */
    private function initializeComponents(): void {
        wp_mkdir_p($this->config['temp_dir']);
        
        $this->db_importer = new DatabaseImporter();
        $this->plugin_reconstructor = new PluginReconstructor();
        $this->file_restorer = new FileRestorer();
        $this->url_migrator = new UrlMigrator();
        $this->bricks_handler = new BricksHandler();
        $this->builder_handler = new BuilderHandler();
        $this->chunk_processor = new ChunkProcessor(
            'restore_' . time(),
            $this->config['temp_dir']
        );
    }
    
    /**
     * Leert alle Caches
     */
    private function clearAllCaches(): void {
        // WordPress Object Cache
        wp_cache_flush();
        
        // Alle Bricks-Options aus dem Cache entfernen
        $bricks_options = [
            'bricks_global_classes', 'bricks_global_elements', 'bricks_global_settings',
            'bricks_color_palette', 'bricks_color_palettes', 'bricks_breakpoints',
            'bricks_font_face_rules', 'bricks_remote_templates', 'bricks_sidebars',
        ];
        foreach ($bricks_options as $opt) {
            wp_cache_delete($opt, 'options');
        }
        wp_cache_delete('alloptions', 'options');
        wp_cache_delete('notoptions', 'options');
        
        // Transients löschen
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_site_transient_%'");
        
        // KRITISCH: Bricks-spezifische Transients löschen
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%bricks%'  AND option_name LIKE '%transient%'");
        
        // Bekannte Cache-Plugins
        if (function_exists('wp_cache_clear_cache')) {
            wp_cache_clear_cache(); // WP Super Cache
        }
        
        if (function_exists('w3tc_flush_all')) {
            w3tc_flush_all(); // W3 Total Cache
        }
        
        if (function_exists('rocket_clean_domain')) {
            rocket_clean_domain(); // WP Rocket
        }
        
        // KRITISCH: Bricks CSS-Cache löschen
        $this->clearBricksCssCache();
        
        // Cache-Verzeichnisse leeren
        $cache_dirs = [
            WP_CONTENT_DIR . '/cache/',
            WP_CONTENT_DIR . '/wp-cache/',
        ];
        
        foreach ($cache_dirs as $dir) {
            if (is_dir($dir)) {
                $this->recursiveDelete($dir, false);
            }
        }
    }
    
    /**
     * Löscht Bricks CSS-Cache und erzwingt Neugenerierung
     */
    private function clearBricksCssCache(): void {
        // Bricks CSS-Verzeichnis
        $bricks_css_dir = WP_CONTENT_DIR . '/uploads/bricks/css/';
        
        if (is_dir($bricks_css_dir)) {
            $this->recursiveDelete($bricks_css_dir, false);
            
        }
        
        // Bricks Asset-Verzeichnis
        $bricks_assets_dir = WP_CONTENT_DIR . '/uploads/bricks/assets/';
        if (is_dir($bricks_assets_dir)) {
            $this->recursiveDelete($bricks_assets_dir, false);
        }
        
        // Bricks Temp-Verzeichnis
        $bricks_temp_dir = WP_CONTENT_DIR . '/uploads/bricks/temp/';
        if (is_dir($bricks_temp_dir)) {
            $this->recursiveDelete($bricks_temp_dir, false);
        }
        
        // Bricks-spezifische Option die CSS-Regenerierung erzwingt
        global $wpdb;
        
        // Lösche Bricks CSS-Version um Neugenerierung zu erzwingen
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE 'bricks_css_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE 'bricks_post_%'");
        
        // Lösche Bricks-generierte postmeta CSS-Hashes
        $wpdb->query("DELETE FROM {$wpdb->postmeta} WHERE meta_key = '_bricks_css_version'");
        $wpdb->query("DELETE FROM {$wpdb->postmeta} WHERE meta_key = '_bricks_page_content_2_hash'");
    }
    
    /**
     * Löscht Verzeichnis-Inhalt rekursiv
     */
    private function recursiveDelete(string $dir, bool $delete_root = true): void {
        if (!is_dir($dir)) {
            return;
        }
        
        $files = array_diff(scandir($dir), ['.', '..']);
        
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            
            if (is_dir($path)) {
                $this->recursiveDelete($path, true);
            } else {
                @unlink($path);
            }
        }
        
        if ($delete_root) {
            @rmdir($dir);
        }
    }
    
    /**
     * Meldet Fortschritt
     */
    private function reportProgress(string $phase, float $percent, string $message): void {
        if ($this->progress_callback) {
            call_user_func($this->progress_callback, $phase, $percent, $message);
        }
    }
    
    /**
     * Getter für Container
     */
    public function getContainer(): Container {
        return $this->container;
    }
    
    /**
     * Getter für Stats
     */
    public function getStats(): array {
        return $this->stats;
    }
    
    /**
     * Gibt eine Vorschau des Restores zurück
     * 
     * @return array
     */
    public function preview(): array {
        $manifest = $this->container->getManifest();
        
        return [
            'backup_type' => $manifest->get('backup_type'),
            'created_at' => $manifest->get('created_at'),
            'wordpress_version' => $manifest->get('wordpress')['version'] ?? 'unknown',
            'site_url' => $manifest->getSiteUrl(),
            'current_url' => $this->original_site_url,
            'url_migration_needed' => $manifest->getSiteUrl() !== $this->original_site_url,
            'database' => $manifest->get('database'),
            'plugins' => count($manifest->getPlugins()),
            'segments' => count($manifest->getSegments()),
            'stats' => $manifest->get('stats'),
        ];
    }
}

