<?php
/**
 * ChunkProcessor - Batch/Chunk-Verarbeitung für große Operationen
 * 
 * Verarbeitet große Datenmengen in kleinen Batches, um:
 * - PHP Execution Timeout zu vermeiden
 * - Memory Limit einzuhalten
 * - Fortschritt speicherbar zu machen (Checkpoints)
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Core;

if (!defined('ABSPATH')) {
    exit;
}

class ChunkProcessor {
    
    /** @var int Standard-Batch-Größe */
    const DEFAULT_BATCH_SIZE = 100;
    
    /** @var int Max. Ausführungszeit pro Batch (Sekunden) */
    const MAX_EXECUTION_TIME = 25;
    
    /** @var int Max. Memory-Nutzung (80% des Limits) */
    private $max_memory;
    
    /** @var int Start-Zeit */
    private $start_time;
    
    /** @var string State-Datei für Checkpoints */
    private $state_file;
    
    /** @var array Aktueller State */
    private $state = [];
    
    /** @var callable|null Progress-Callback */
    private $progress_callback;
    
    /**
     * Erstellt einen neuen ChunkProcessor
     * 
     * @param string $job_id Eindeutige Job-ID für Checkpoints
     * @param string $state_dir Verzeichnis für State-Dateien
     */
    public function __construct(string $job_id, string $state_dir) {
        $this->state_file = trailingslashit($state_dir) . "chunk_state_{$job_id}.json";
        $this->max_memory = $this->calculateMaxMemory();
        $this->start_time = time();
        
        // Bestehenden State laden falls vorhanden
        if (file_exists($this->state_file)) {
            $this->state = json_decode(file_get_contents($this->state_file), true) ?? [];
        }
    }
    
    /**
     * Setzt den Progress-Callback
     * 
     * @param callable $callback function($processed, $total, $message)
     */
    public function setProgressCallback(callable $callback): void {
        $this->progress_callback = $callback;
    }
    
    /**
     * Verarbeitet ein Array in Chunks
     * 
     * @param array $items Zu verarbeitende Items
     * @param callable $processor Verarbeitungsfunktion pro Item
     * @param int $batch_size Batch-Größe
     * @return array Ergebnis mit Status
     */
    public function processArray(array $items, callable $processor, int $batch_size = self::DEFAULT_BATCH_SIZE): array {
        $total = count($items);
        $processed = $this->state['processed'] ?? 0;
        $results = $this->state['results'] ?? [];
        $errors = $this->state['errors'] ?? [];
        
        // Von letztem Checkpoint fortsetzen
        $items = array_slice($items, $processed, null, true);
        
        foreach (array_chunk($items, $batch_size, true) as $batch) {
            // Ressourcen-Check
            if ($this->shouldStop()) {
                $this->saveState($processed, $results, $errors, false);
                return [
                    'complete' => false,
                    'processed' => $processed,
                    'total' => $total,
                    'results' => $results,
                    'errors' => $errors,
                    'message' => 'Checkpoint gespeichert, bitte fortsetzen',
                ];
            }
            
            // Batch verarbeiten
            foreach ($batch as $key => $item) {
                try {
                    $result = $processor($item, $key);
                    if ($result !== null) {
                        $results[$key] = $result;
                    }
                } catch (\Exception $e) {
                    $errors[$key] = $e->getMessage();
                }
                
                $processed++;
                
                // Progress melden
                if ($this->progress_callback) {
                    call_user_func($this->progress_callback, $processed, $total, "Verarbeite $processed von $total");
                }
            }
            
            // Speicher freigeben
            $this->freeMemory();
        }
        
        // Abgeschlossen - State löschen
        $this->clearState();
        
        return [
            'complete' => true,
            'processed' => $processed,
            'total' => $total,
            'results' => $results,
            'errors' => $errors,
        ];
    }
    
    /**
     * Verarbeitet Dateien in Chunks
     * 
     * @param array $files Datei-Liste
     * @param callable $processor Verarbeitungsfunktion pro Datei
     * @param int $batch_size Batch-Größe
     * @return array
     */
    public function processFiles(array $files, callable $processor, int $batch_size = 50): array {
        return $this->processArray($files, $processor, $batch_size);
    }
    
    /**
     * Verarbeitet Datenbank-Zeilen in Chunks
     * 
     * @param string $table Tabellen-Name
     * @param callable $processor Verarbeitungsfunktion
     * @param int $batch_size Batch-Größe
     * @param string $order_by Sortierung
     * @return array
     */
    public function processTable(string $table, callable $processor, int $batch_size = 500, string $order_by = 'id'): array {
        global $wpdb;
        
        $offset = $this->state['offset'] ?? 0;
        $processed = $this->state['processed'] ?? 0;
        $results = $this->state['results'] ?? [];
        $errors = $this->state['errors'] ?? [];
        
        // Gesamtanzahl ermitteln
        $total = (int) $wpdb->get_var("SELECT COUNT(*) FROM `$table`");
        
        while ($offset < $total) {
            // Ressourcen-Check
            if ($this->shouldStop()) {
                $this->saveState($processed, $results, $errors, false, $offset);
                return [
                    'complete' => false,
                    'processed' => $processed,
                    'total' => $total,
                    'offset' => $offset,
                    'results' => $results,
                    'errors' => $errors,
                ];
            }
            
            // Batch laden
            $rows = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM `$table` ORDER BY `$order_by` LIMIT %d OFFSET %d",
                    $batch_size,
                    $offset
                ),
                ARRAY_A
            );
            
            if (empty($rows)) {
                break;
            }
            
            // Batch verarbeiten
            foreach ($rows as $row) {
                try {
                    $result = $processor($row, $table);
                    if ($result !== null) {
                        $results[] = $result;
                    }
                } catch (\Exception $e) {
                    $errors[] = [
                        'table' => $table,
                        'row' => $row,
                        'error' => $e->getMessage(),
                    ];
                }
                
                $processed++;
            }
            
            $offset += $batch_size;
            
            // Progress melden
            if ($this->progress_callback) {
                call_user_func($this->progress_callback, $processed, $total, "Tabelle $table: $processed von $total");
            }
            
            // Speicher freigeben
            $this->freeMemory();
        }
        
        // Abgeschlossen
        $this->clearState();
        
        return [
            'complete' => true,
            'processed' => $processed,
            'total' => $total,
            'results' => $results,
            'errors' => $errors,
        ];
    }
    
    /**
     * Führt SQL-Statements in Chunks aus
     * 
     * @param array $statements SQL-Statements
     * @param int $batch_size Statements pro Batch
     * @return array
     */
    public function executeSqlStatements(array $statements, int $batch_size = 50): array {
        global $wpdb;
        
        $total = count($statements);
        $processed = $this->state['processed'] ?? 0;
        $successful = $this->state['successful'] ?? 0;
        $errors = $this->state['errors'] ?? [];
        
        // Von Checkpoint fortsetzen
        $statements = array_slice($statements, $processed);
        
        foreach (array_chunk($statements, $batch_size) as $batch) {
            if ($this->shouldStop()) {
                $this->saveState($processed, ['successful' => $successful], $errors, false);
                return [
                    'complete' => false,
                    'processed' => $processed,
                    'total' => $total,
                    'successful' => $successful,
                    'errors' => $errors,
                ];
            }
            
            foreach ($batch as $sql) {
                $result = $wpdb->query($sql);
                
                if ($result === false) {
                    $errors[] = [
                        'sql' => substr($sql, 0, 200) . '...',
                        'error' => $wpdb->last_error,
                    ];
                } else {
                    $successful++;
                }
                
                $processed++;
            }
            
            // Progress melden
            if ($this->progress_callback) {
                call_user_func($this->progress_callback, $processed, $total, "SQL: $processed von $total");
            }
        }
        
        $this->clearState();
        
        return [
            'complete' => true,
            'processed' => $processed,
            'total' => $total,
            'successful' => $successful,
            'errors' => $errors,
        ];
    }
    
    /**
     * Prüft ob ein Checkpoint existiert
     * 
     * @return bool
     */
    public function hasCheckpoint(): bool {
        return file_exists($this->state_file) && !empty($this->state);
    }
    
    /**
     * Löscht den Checkpoint
     */
    public function clearState(): void {
        if (file_exists($this->state_file)) {
            unlink($this->state_file);
        }
        $this->state = [];
    }
    
    /**
     * Gibt den aktuellen Fortschritt zurück
     * 
     * @return array
     */
    public function getProgress(): array {
        return $this->state;
    }
    
    // ========================================
    // PRIVATE HELPER
    // ========================================
    
    /**
     * Prüft ob die Verarbeitung pausiert werden sollte
     */
    private function shouldStop(): bool {
        // Zeit-Check
        $elapsed = time() - $this->start_time;
        if ($elapsed >= self::MAX_EXECUTION_TIME) {
            return true;
        }
        
        // Memory-Check
        if (memory_get_usage(true) >= $this->max_memory) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Speichert den aktuellen State
     */
    private function saveState(int $processed, array $results, array $errors, bool $complete, int $offset = 0): void {
        $this->state = [
            'processed' => $processed,
            'results' => $results,
            'errors' => $errors,
            'complete' => $complete,
            'offset' => $offset,
            'saved_at' => time(),
        ];
        
        file_put_contents($this->state_file, json_encode($this->state));
    }
    
    /**
     * Berechnet das maximale Memory-Limit
     */
    private function calculateMaxMemory(): int {
        $limit = ini_get('memory_limit');
        
        if ($limit === '-1') {
            return PHP_INT_MAX; // Unlimited
        }
        
        // Parse (z.B. "256M" -> Bytes)
        $limit = trim($limit);
        $last = strtolower($limit[strlen($limit) - 1]);
        $limit = (int) $limit;
        
        switch ($last) {
            case 'g':
                $limit *= 1024;
            case 'm':
                $limit *= 1024;
            case 'k':
                $limit *= 1024;
        }
        
        // 80% des Limits nutzen
        return (int) ($limit * 0.8);
    }
    
    /**
     * Gibt Speicher frei
     */
    private function freeMemory(): void {
        if (function_exists('gc_collect_cycles')) {
            gc_collect_cycles();
        }
        
        // WordPress Object Cache leeren
        wp_cache_flush();
    }
    
    /**
     * Statische Hilfsmethode: Zeit-Limit erweitern
     */
    public static function extendTimeLimit(int $seconds = 60): void {
        if (!ini_get('safe_mode')) {
            @set_time_limit($seconds);
        }
    }
    
    /**
     * Statische Hilfsmethode: Memory-Limit erhöhen
     */
    public static function increaseMemory(string $limit = '512M'): void {
        @ini_set('memory_limit', $limit);
    }
}

