<?php
/**
 * Container - Das .jbm Backup-Format
 * 
 * Ein JBM-Container ist ein Verzeichnis mit:
 * - manifest.json (Metadaten, Checksums, Dateiliste)
 * - database.sql.gz (Bereinigte, komprimierte DB)
 * - plugins.json (Plugin-Inventar)
 * - segments/ (Verzeichnis mit ZIP-Segmenten)
 * 
 * Der Container kann als einzelne .jbm Datei (ZIP) exportiert werden.
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Core;

if (!defined('ABSPATH')) {
    exit;
}

class Container {
    
    /** @var string Container-Pfad */
    private $path;
    
    /** @var Manifest */
    private $manifest;
    
    /** @var array<Segment> Aktive Segmente */
    private $segments = [];
    
    /** @var array Segment-Counter pro Typ */
    private $segment_counters = [];
    
    /** @var bool Ob Container geöffnet ist */
    private $is_open = false;
    
    /**
     * Erstellt einen neuen Container
     * 
     * @param string $path Container-Pfad (Verzeichnis)
     * @param string $backup_type Backup-Typ
     */
    public function __construct(string $path, string $backup_type = 'full') {
        $this->path = trailingslashit($path);
        $this->manifest = new Manifest($backup_type);
    }
    
    /**
     * Öffnet einen bestehenden Container
     * 
     * @param string $path Pfad zum Container oder .jbm Datei
     * @return self
     * @throws \Exception
     */
    public static function open(string $path): self {
        // Prüfen ob es eine .jbm Datei ist (ZIP)
        if (pathinfo($path, PATHINFO_EXTENSION) === 'jbm') {
            $path = self::extractJbmFile($path);
        }
        
        $container = new self($path);
        $container->loadManifest();
        $container->is_open = true;
        
        return $container;
    }
    
    /**
     * Erstellt den Container (Verzeichnisstruktur)
     * 
     * @return bool
     */
    public function create(): bool {
        // Hauptverzeichnis erstellen
        if (!wp_mkdir_p($this->path)) {
            throw new \Exception("Konnte Container-Verzeichnis nicht erstellen: {$this->path}");
        }
        
        // Segment-Verzeichnis erstellen
        if (!wp_mkdir_p($this->path . 'segments')) {
            throw new \Exception("Konnte Segment-Verzeichnis nicht erstellen");
        }
        
        $this->is_open = true;
        return true;
    }
    
    /**
     * Fügt die Datenbank zum Container hinzu
     * 
     * @param string $sql_content SQL-Inhalt
     * @param array $tables Tabellen-Info
     * @param array $cleanup_stats Bereinigungsstatistiken
     * @return bool
     */
    public function addDatabase(string $sql_content, array $tables, array $cleanup_stats = []): bool {
        $db_path = $this->path . 'database.sql';
        $gz_path = $db_path . '.gz';
        
        // SQL komprimieren
        $gz = gzopen($gz_path, 'w9');
        if (!$gz) {
            throw new \Exception("Konnte Datenbank nicht komprimieren");
        }
        
        gzwrite($gz, $sql_content);
        gzclose($gz);
        
        // Manifest aktualisieren
        $this->manifest->setDatabaseInfo($tables, $cleanup_stats);
        $this->manifest->addSegment(
            'database',
            'database',
            filesize($gz_path),
            hash_file('sha256', $gz_path)
        );
        
        return true;
    }
    
    /**
     * Fügt eine Datei zum Container hinzu
     * 
     * @param string $source_path Quellpfad
     * @param string $type Typ ('uploads', 'themes', 'plugins')
     * @param string $relative_path Relativer Pfad im Archiv
     * @return bool
     */
    public function addFile(string $source_path, string $type, string $relative_path): bool {
        $segment = $this->getOrCreateSegment($type);
        
        $result = $segment->addFile($source_path, $relative_path);
        
        // Wenn Segment voll, neues erstellen
        if (!$result && $segment->isFull()) {
            $this->closeSegment($type);
            $segment = $this->getOrCreateSegment($type);
            $result = $segment->addFile($source_path, $relative_path);
        }
        
        if ($result) {
            $this->manifest->addFile(
                $relative_path,
                filesize($source_path),
                hash_file('sha256', $source_path),
                $segment->getName()
            );
        }
        
        return $result;
    }
    
    /**
     * Fügt String-Inhalt als Datei hinzu
     * 
     * @param string $content Inhalt
     * @param string $type Typ
     * @param string $archive_path Pfad im Archiv
     * @return bool
     */
    public function addFromString(string $content, string $type, string $archive_path): bool {
        $segment = $this->getOrCreateSegment($type);
        
        $result = $segment->addFromString($content, $archive_path);
        
        if ($result) {
            $this->manifest->addFile(
                $archive_path,
                strlen($content),
                hash('sha256', $content),
                $segment->getName()
            );
        }
        
        return $result;
    }
    
    /**
     * Fügt Plugin zum Inventar hinzu
     * 
     * @param array $plugin_data Plugin-Daten
     */
    public function addPlugin(array $plugin_data): void {
        $this->manifest->addPlugin($plugin_data);
    }
    
    /**
     * Schließt und finalisiert den Container
     * 
     * @return array Container-Info
     */
    public function close(): array {
        // Alle offenen Segmente schließen
        foreach ($this->segments as $type => $segment) {
            $info = $segment->close();
            $this->manifest->addSegment(
                $info['name'],
                $info['type'],
                $info['size'],
                $info['checksum']
            );
        }
        
        // Plugin-Inventar speichern
        $plugins_json = json_encode($this->manifest->getPlugins(), JSON_PRETTY_PRINT);
        file_put_contents($this->path . 'plugins.json', $plugins_json);
        
        // Manifest speichern
        $this->manifest->save($this->path . 'manifest.json');
        
        // Gesamtgröße berechnen
        $total_size = $this->calculateTotalSize();
        $this->manifest->setCompressedSize($total_size);
        $this->manifest->save($this->path . 'manifest.json');
        
        $this->is_open = false;
        
        return [
            'path' => $this->path,
            'manifest' => $this->manifest->toArray(),
            'total_size' => $total_size,
        ];
    }
    
    /**
     * Exportiert Container als einzelne .jbm Datei
     * 
     * @param string $output_path Zielpfad für .jbm Datei
     * @return string Pfad zur erstellten Datei
     */
    public function export(string $output_path): string {
        if ($this->is_open) {
            $this->close();
        }
        
        // Timeout erhöhen für große Backups
        @set_time_limit(600); // 10 Minuten
        @ini_set('max_execution_time', '600');
        
        
        $zip = new \ZipArchive();
        $result = $zip->open($output_path, \ZipArchive::CREATE | \ZipArchive::OVERWRITE);
        
        if ($result !== true) {
            throw new \Exception("Konnte .jbm Datei nicht erstellen (Error: $result)");
        }
        
        // Alle Dateien im Container hinzufügen
        $this->addDirectoryToZip($zip, $this->path, '');
        
        $num_files = $zip->numFiles;
        
        
        $zip->close();
        
        
        return $output_path;
    }
    
    /**
     * Getter für Manifest
     */
    public function getManifest(): Manifest {
        return $this->manifest;
    }
    
    /**
     * Getter für Pfad
     */
    public function getPath(): string {
        return $this->path;
    }
    
    // ========================================
    // RESTORE-METHODEN
    // ========================================
    
    /**
     * Gibt das Datenbank-Segment zurück
     * 
     * @return string|null SQL-Inhalt oder null
     */
    public function getDatabase(): ?string {
        $gz_path = $this->path . 'database.sql.gz';
        
        if (!file_exists($gz_path)) {
            return null;
        }
        
        $content = '';
        $gz = gzopen($gz_path, 'r');
        
        while (!gzeof($gz)) {
            $content .= gzread($gz, 8192);
        }
        
        gzclose($gz);
        
        return $content;
    }
    
    /**
     * Gibt ein Segment zum Lesen zurück
     * 
     * @param string $segment_name Segment-Name
     * @return Segment
     */
    public function getSegment(string $segment_name): Segment {
        $segments = $this->manifest->getSegments();
        
        if (!isset($segments[$segment_name])) {
            throw new \Exception("Segment nicht gefunden: $segment_name");
        }
        
        $info = $segments[$segment_name];
        $segment = new Segment(
            $segment_name,
            $info['type'],
            $this->path . 'segments'
        );
        
        return $segment;
    }
    
    /**
     * Extrahiert alle Segmente eines Typs
     * 
     * @param string $type Segment-Typ
     * @param string $destination Zielverzeichnis
     * @param callable|null $progress Fortschritts-Callback
     * @return int Anzahl extrahierter Dateien
     */
    public function extractType(string $type, string $destination, ?callable $progress = null): int {
        $segments = $this->manifest->getSegments();
        $total = 0;
        
        foreach ($segments as $name => $info) {
            if ($info['type'] === $type) {
                $segment = $this->getSegment($name);
                $segment->openForReading();
                $total += $segment->extractAll($destination, $progress);
                $segment->close();
            }
        }
        
        return $total;
    }
    
    /**
     * Verifiziert Container-Integrität
     * 
     * @return array Verifizierungsergebnis
     */
    public function verify(): array {
        $result = [
            'valid' => true,
            'manifest_valid' => false,
            'segments' => [],
            'errors' => [],
        ];
        
        // Manifest prüfen
        $result['manifest_valid'] = $this->manifest->verify();
        if (!$result['manifest_valid']) {
            $result['valid'] = false;
            $result['errors'][] = 'Manifest-Checksumme ungültig';
        }
        
        // Segmente prüfen
        $segments = $this->manifest->getSegments();
        foreach ($segments as $name => $info) {
            // Datenbank wird im Root als database.sql.gz gespeichert
            if ($name === 'database') {
                $segment_path = $this->path . 'database.sql.gz';
            } else {
                $segment_path = $this->path . 'segments/' . $name . '.zip';
            }
            
            if (!file_exists($segment_path)) {
                $result['valid'] = false;
                $result['segments'][$name] = 'FEHLT';
                $result['errors'][] = "Segment fehlt: $name";
                continue;
            }
            
            $actual_checksum = hash_file('sha256', $segment_path);
            if (!hash_equals($info['checksum'], $actual_checksum)) {
                $result['valid'] = false;
                $result['segments'][$name] = 'KORRUPT';
                $result['errors'][] = "Segment korrupt: $name";
            } else {
                $result['segments'][$name] = 'OK';
            }
        }
        
        return $result;
    }
    
    // ========================================
    // PRIVATE HELPER
    // ========================================
    
    /**
     * Lädt das Manifest
     */
    private function loadManifest(): void {
        $manifest_path = $this->path . 'manifest.json';
        
        if (!file_exists($manifest_path)) {
            throw new \Exception("Manifest nicht gefunden: $manifest_path");
        }
        
        $this->manifest = Manifest::load($manifest_path);
    }
    
    /**
     * Holt oder erstellt ein Segment
     */
    private function getOrCreateSegment(string $type): Segment {
        if (!isset($this->segments[$type]) || $this->segments[$type]->isFull()) {
            // Altes Segment schließen wenn vorhanden
            if (isset($this->segments[$type])) {
                $this->closeSegment($type);
            }
            
            // Counter erhöhen
            if (!isset($this->segment_counters[$type])) {
                $this->segment_counters[$type] = 0;
            }
            $this->segment_counters[$type]++;
            
            // Neues Segment erstellen
            $name = sprintf('%s-%03d', $type, $this->segment_counters[$type]);
            $this->segments[$type] = new Segment($name, $type, $this->path . 'segments');
            $this->segments[$type]->open();
        }
        
        return $this->segments[$type];
    }
    
    /**
     * Schließt ein Segment
     */
    private function closeSegment(string $type): void {
        if (isset($this->segments[$type])) {
            $info = $this->segments[$type]->close();
            $this->manifest->addSegment(
                $info['name'],
                $info['type'],
                $info['size'],
                $info['checksum']
            );
            unset($this->segments[$type]);
        }
    }
    
    /**
     * Berechnet die Gesamtgröße des Containers
     */
    private function calculateTotalSize(): int {
        $total = 0;
        
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($this->path, \RecursiveDirectoryIterator::SKIP_DOTS)
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $total += $file->getSize();
            }
        }
        
        return $total;
    }
    
    /**
     * Fügt ein Verzeichnis rekursiv zu einem ZIP hinzu
     */
    private function addDirectoryToZip(\ZipArchive $zip, string $dir, string $prefix): void {
        $files = @scandir($dir);
        
        if ($files === false) {
            return;
        }
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }
            
            $path = $dir . $file;
            $archive_path = $prefix . $file;
            
            if (is_dir($path)) {
                $zip->addEmptyDir($archive_path);
                $this->addDirectoryToZip($zip, $path . '/', $archive_path . '/');
            } else {
                $zip->addFile($path, $archive_path);
            }
        }
    }
    
    /**
     * Berechnet die Größe eines Verzeichnisses
     */
    private function getDirectorySize(string $dir): int {
        $size = 0;
        $files = @scandir($dir);
        
        if ($files === false) {
            return 0;
        }
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }
            
            $path = $dir . $file;
            
            if (is_dir($path)) {
                $size += $this->getDirectorySize($path . '/');
            } else {
                $size += @filesize($path) ?: 0;
            }
        }
        
        return $size;
    }
    
    /**
     * Extrahiert eine .jbm Datei
     */
    private static function extractJbmFile(string $jbm_path): string {
        $extract_dir = dirname($jbm_path) . '/' . pathinfo($jbm_path, PATHINFO_FILENAME) . '_extracted/';
        
        $zip = new \ZipArchive();
        $result = $zip->open($jbm_path);
        
        if ($result !== true) {
            throw new \Exception("Konnte .jbm Datei nicht öffnen: $jbm_path");
        }
        
        $zip->extractTo($extract_dir);
        $zip->close();
        
        return $extract_dir;
    }
}

